/*
 * Java
 *
 * Copyright 2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.event;

import java.io.IOException;

/**
 * An EventDataReader contains the methods to read the data of an extended event.
 * <p>
 * EventDataReaders are created by the system and passed to extended events listeners when calling
 * {@link EventQueueListener#handleExtendedEvent(int, EventDataReader)} method.
 * <p>
 * There are two ways to read the data of an extended event:
 * <ul>
 * <li>Read the data with {@link EventDataReader#read(byte[], int, int)} or {@link EventDataReader#readFully(byte[])}
 * methods.
 * <ul>
 * <li>You will get the data in a byte array and can process it on your own in
 * {@link EventQueueListener#handleExtendedEvent(int, EventDataReader)}.</li>
 * </ul>
 * </li>
 * <li>Read the data with the methods related to the primitive types such as {@link EventDataReader#readBoolean()} or
 * {@link EventDataReader#readByte()}.
 * <ul>
 * <li>This is useful when the extended data you send through the Event Queue is a C structure with multiple
 * fields.</li>
 * <li>To use the methods, <b>your fields must follow this alignment</b>:
 * <ul>
 * <li>A boolean (1 byte) will be 1-byte aligned.</li>
 * <li>A byte (1 byte) will be 1-byte aligned.</li>
 * <li>A char (2 bytes) will be 2-byte aligned.</li>
 * <li>A double (8 bytes) will be 8-byte aligned.</li>
 * <li>A float (4 bytes) will be 4-byte aligned.</li>
 * <li>An int (4 bytes) will be 4-byte aligned.</li>
 * <li>A long (8 bytes) will be 8-byte aligned.</li>
 * <li>A short (2 bytes) will be 2-byte aligned.</li>
 * <li>An unsigned byte (1 byte) will be 1-byte aligned.</li>
 * <li>A unsigned short (2 bytes) will be 2-byte aligned.</li>
 * </ul>
 * </li>
 * </ul>
 * </li>
 * </ul>
 *
 */
public interface EventDataReader {

	/**
	 * Reads the next boolean of data.
	 *
	 * @return the next boolean of data.
	 * @throws IOException
	 *             if there is no boolean remaining in the data of the extended event.
	 */
	boolean readBoolean() throws IOException;

	/**
	 * Reads the next byte of data.
	 *
	 * @return the next byte of data.
	 * @throws IOException
	 *             if there is no byte remaining in the data of the extended event.
	 */
	byte readByte() throws IOException;

	/**
	 * Reads the next char of data.
	 *
	 * @return the next char of data.
	 * @throws IOException
	 *             if there is no char remaining in the data of the extended event.
	 */
	char readChar() throws IOException;

	/**
	 * Reads the next double of data.
	 *
	 * @return the next double of data.
	 * @throws IOException
	 *             if there is no double remaining in the data of the extended event.
	 */
	double readDouble() throws IOException;

	/**
	 * Reads the next float of data.
	 *
	 * @return the next float of data.
	 * @throws IOException
	 *             if there is no float remaining in the data of the extended event.
	 */
	float readFloat() throws IOException;

	/**
	 * Reads all the bytes of data and store them into the buffer {@code b}.
	 *
	 * @param b
	 *            the buffer into which the data is read.
	 * @return the total number of bytes read into the buffer.
	 * @throws IOException
	 *             if there is no byte remaining in the data of the extended event or if the buffer is too small to
	 *             store the event data.
	 */
	int readFully(byte[] b) throws IOException;

	/**
	 * Reads {@code len} bytes of data and store them into the buffer {@code b}.
	 *
	 * @param b
	 *            the buffer into which the data is read.
	 * @param off
	 *            the start offset in the buffer {@code b} at which the data is written.
	 * @param len
	 *            the number of bytes to read.
	 * @return the total number of bytes read into the buffer.
	 * @throws IOException
	 *             if there are not enough bytes remaining in the data of the extended event or if the buffer is too
	 *             small to store the event data.
	 */
	int read(byte[] b, int off, int len) throws IOException;

	/**
	 * Reads the next integer of data.
	 *
	 * @return the next integer of data.
	 * @throws IOException
	 *             if there is no integer remaining in the data of the extended event.
	 */
	int readInt() throws IOException;

	/**
	 * Reads the next long of data.
	 *
	 * @return the next long of data.
	 * @throws IOException
	 *             if there is no long remaining in the data of the extended event.
	 */
	long readLong() throws IOException;

	/**
	 * Reads the next short of data.
	 *
	 * @return the next short of data.
	 * @throws IOException
	 *             if there is no short remaining in the data of the extended event.
	 */
	short readShort() throws IOException;

	/**
	 * Reads the next unsigned byte of data.
	 *
	 * @return the next unsigned byte of data.
	 * @throws IOException
	 *             if there is no unsigned byte remaining in the data of the extended event.
	 */
	int readUnsignedByte() throws IOException;

	/**
	 * Reads the next unsigned short of data.
	 *
	 * @return the next unsigned short of data.
	 * @throws IOException
	 *             if there is no unsigned short remaining in the data of the extended event.
	 */
	int readUnsignedShort() throws IOException;

	/**
	 * Skips over and discard {@code n} bytes of data.
	 *
	 * @param n
	 *            the number of bytes to skip.
	 * @return 0 if OK and -1 if an error occurs.
	 */
	int skipBytes(int n);

	/**
	 * Returns the number of bytes that can be read (or skipped over).
	 *
	 * @return the number of available data bytes.
	 */
	int available();

}
