/*
 * Copyright 2015-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.style.outline.border;

import ej.annotation.Nullable;
import ej.bon.XMath;
import ej.drawing.ShapePainter;
import ej.drawing.ShapePainter.Cap;
import ej.microui.display.GraphicsContext;
import ej.mwt.style.outline.Outline;
import ej.mwt.util.Outlineable;
import ej.mwt.util.Size;

/**
 * Draws a uniform border with round corners.
 * <p>
 * May be associated with {@link ej.mwt.style.background.RoundedBackground} or used on its own.
 * <p>
 * The thickness and corner radius are stored as a <code>char</code> for heap optimization and therefore cannot exceed
 * <code>65535</code>.
 */
public class RoundedBorder implements Outline {

	private static final int EMPTY_CIRCLE = 0;
	private static final int QUARTER_CIRCLE = 90;
	private static final int HALF_CIRCLE = 180;
	private static final int THREE_QUARTER_CIRCLE = 270;

	private static final int FADE = 1;
	private static final Cap CAP = Cap.NONE;

	private final int color;
	private final char thickness;
	private final char cornerRadius;

	/**
	 * Creates a rounded border specifying its attributes.
	 * <p>
	 * The given corner radius and thickness are clamped between <code>0</code> and <code>Character.MAX_VALUE</code>.
	 *
	 * @param color
	 *            the color.
	 * @param cornerRadius
	 *            the corner radius.
	 * @param thickness
	 *            the border thickness.
	 */
	public RoundedBorder(int color, int cornerRadius, int thickness) {
		this.color = color;
		this.thickness = (char) XMath.limit(thickness, 0, Character.MAX_VALUE);
		this.cornerRadius = (char) XMath.limit(cornerRadius, 0, Character.MAX_VALUE);
	}

	/**
	 * Gets the color.
	 *
	 * @return the color.
	 */
	public int getColor() {
		return this.color;
	}

	/**
	 * Gets the thickness.
	 *
	 * @return the thickness.
	 */
	public int getThickness() {
		return this.thickness;
	}

	/**
	 * Gets the corner radius.
	 *
	 * @return the corner radius.
	 */
	public int getCornerRadius() {
		return this.cornerRadius;
	}

	@Override
	public void apply(Outlineable outlineable) {
		int thickness = this.thickness;
		outlineable.removeOutline(thickness, thickness, thickness, thickness);
	}

	@Override
	public void apply(GraphicsContext g, Size size) {
		boolean hasBackgroundColor = g.hasBackgroundColor();
		int backgroundColor = g.getBackgroundColor();
		g.removeBackgroundColor();

		int width = size.getWidth();
		int height = size.getHeight();
		int thickness = this.thickness;

		// Both radius have the same size.
		int cornerRadius = Math.min(this.cornerRadius, (Math.min(width, height) >> 1) - thickness);

		g.setColor(this.color);

		int odd = thickness & 0x1;
		int notOdd = 1 - odd;

		int borderDiameter = (cornerRadius << 1);
		int halfThickness = thickness >> 1;
		int leftX = halfThickness + odd;
		int topY = halfThickness + odd;
		int rightX = leftX + width - thickness - 1 - 2 * odd;
		int bottomY = topY + height - thickness - 1 - 2 * odd;

		ShapePainter.drawThickFadedCircleArc(g, leftX, topY, borderDiameter, QUARTER_CIRCLE, QUARTER_CIRCLE, thickness,
				FADE, CAP, CAP);
		ShapePainter.drawThickFadedCircleArc(g, leftX, bottomY - borderDiameter - notOdd, borderDiameter, HALF_CIRCLE,
				QUARTER_CIRCLE, thickness, FADE, CAP, CAP);
		ShapePainter.drawThickFadedCircleArc(g, rightX - borderDiameter - notOdd, bottomY - borderDiameter - notOdd,
				borderDiameter, THREE_QUARTER_CIRCLE, QUARTER_CIRCLE, thickness, FADE, CAP, CAP);
		ShapePainter.drawThickFadedCircleArc(g, rightX - borderDiameter - notOdd, topY, borderDiameter, EMPTY_CIRCLE,
				QUARTER_CIRCLE, thickness, FADE, CAP, CAP);

		int horizontalLeftX = leftX + cornerRadius;
		int horizontalRightX = rightX - cornerRadius + odd;
		int bottomLineY = bottomY + odd;
		ShapePainter.drawThickFadedLine(g, horizontalLeftX, topY, horizontalRightX, topY, thickness, FADE, CAP, CAP);
		ShapePainter.drawThickFadedLine(g, horizontalLeftX, bottomLineY, horizontalRightX, bottomLineY, thickness, FADE,
				CAP, CAP);

		int verticalTopY = topY + cornerRadius;
		int verticalBottomY = bottomY - cornerRadius + odd;
		int rightLineX = rightX + odd;
		ShapePainter.drawThickFadedLine(g, leftX, verticalTopY, leftX, verticalBottomY, thickness, FADE, CAP, CAP);
		ShapePainter.drawThickFadedLine(g, rightLineX, verticalTopY, rightLineX, verticalBottomY, thickness, FADE, CAP,
				CAP);

		size.removeOutline(thickness, thickness, thickness, thickness);

		g.translate(thickness, thickness);
		g.intersectClip(0, 0, size.getWidth(), size.getHeight());

		if (hasBackgroundColor) {
			g.setBackgroundColor(backgroundColor);
		}
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		if (obj instanceof RoundedBorder) {
			RoundedBorder border = (RoundedBorder) obj;
			return this.color == border.color && this.thickness == border.thickness
					&& this.cornerRadius == border.cornerRadius;
		}
		return false;
	}

	@Override
	public int hashCode() {
		return 17 * this.color + this.thickness * this.cornerRadius;
	}

}
