/*
 * Java
 *
 * Copyright 2020-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import ej.fp.Image;

/**
 * This class provides implementations of native methods of MicroUI library which perform some drawings.
 * <p>
 * The equivalent of embedded structures and enumerations are {@link MicroUIImage}, {@link MicroUIGraphicsContext} and
 * {@link MicroUIImageFormat}. They are used to identify the MicroUI graphical objects: GraphicsContext, Image, image
 * format. etc.
 * <p>
 * Implementation must use the framework object {@link LLUIDisplay#Instance} as monitor to synchronize drawings.
 * <p>
 * All native methods have a byte array as the first parameter. This identifies the destination target: the MicroUI
 * GraphicsContext. This target is retrieved in MicroEJ application calling the method <code>gc.getSNIContext()</code>.
 * The returned byte array must be mapped on a {@link MicroUIGraphicsContext} object in MicroUI native drawing method
 * implementation (see {@link LLUIDisplay#mapMicroUIGraphicsContext(byte[])}).
 * <p>
 * For a MicroUI Image, MicroEJ application has to call <code>image.getSNIContext()</code> to get the image byte array.
 * This byte array must be mapped it on a {@link MicroUIImage} object in MicroUI native drawing method implementation
 * (see {@link LLUIDisplay#mapMicroUIImage(byte[])}).
 * <p>
 * A graphics context holds a clip and the drawer is not allowed to perform a drawing outside this clip (otherwise the
 * behavior is unknown). Note the bottom-right coordinates might be smaller than top-left (in x and/or y) when the clip
 * width and/or height is null. The clip may be disabled (when the current drawing fits the clip); this allows to reduce
 * runtime. See {@link MicroUIGraphicsContext#isClipEnabled()}. Note: several clip methods are available in
 * {@link MicroUIGraphicsContext} to check if a drawing fits the clip.
 * <p>
 * The drawing method must:
 * <ul>
 * <li>Convert the MicroUI graphics context color calling {@link MicroUIGraphicsContext#getRenderingColor()} (in order
 * to respect embedded display pixel representation constraints),</li>
 * <li>Respect the graphics context clip; if not respected, the front panel behavior is unknown,</li>
 * <li>Update the next <code>Display.flush()</code> area (dirty area); if not performed, the next call to
 * <code>Display.flush()</code> will not call
 * {@link LLUIDisplayImpl#flush(MicroUIGraphicsContext, Image, int, int, int, int)} method.</li>
 * </ul>
 *
 * <p>
 * The native method implementation pattern is:
 *
 * <pre>
 * 	void drawXxx(byte[] target, ...) {
 *
 * 		LLUIDisplay graphicalEngine = LLUIDisplay.Instance;
 *
 * 		// all calls to GraphicalEngine.Instance APIs must be synchronized on GraphicalEngine.Instance itself
 * 		synchronized (graphicalEngine) {
 *
 * 			// map a MicroUIGraphicsContext on given byte array which represents the destination target
 * 			MicroUIGraphicsContext gc = graphicalEngine.mapMicroUIGraphicsContext(target);
 *
 * 			// ask to the graphical engine if a drawing can be performed on target
 * 			if (gc.requestDrawing()) {
 *
 *				// call current drawer's equivalent drawing method
 *				getDrawer().drawXxx(gc, x, y);
 *
 *				// contrary to MicroUI native drawing methods implementation for embedded (LLUI_PAINTER_impl.h), there
 *				// is no status to set in MicroUI graphical engine
 *			}
 *		}
 *	}
 * </pre>
 */
public class LLUIPainter {

	/**
	 * Enumerates all MicroUI Image RAW formats.
	 */
	public enum MicroUIImageFormat {
		/**
		 * Defines an image with the same pixel representation and layout as the LCD memory.
		 */
		MICROUI_IMAGE_FORMAT_LCD,

		/**
		 * Defines an image whose pixel format is ARGB8888.
		 */
		MICROUI_IMAGE_FORMAT_ARGB8888,

		/**
		 * Defines an image whose pixel format is RGB888.
		 */
		MICROUI_IMAGE_FORMAT_RGB888,

		/**
		 * Defines an image whose pixel format is RGB565.
		 */
		MICROUI_IMAGE_FORMAT_RGB565,

		/**
		 * Defines an image whose pixel format is ARGB1555.
		 */
		MICROUI_IMAGE_FORMAT_ARGB1555,

		/**
		 * Defines an image whose pixel format is ARGB4444.
		 */
		MICROUI_IMAGE_FORMAT_ARGB4444,

		/**
		 * Defines an image whose pixel format is Alpha1.
		 */
		MICROUI_IMAGE_FORMAT_A1,

		/**
		 * Defines an image whose pixel format is Alpha2.
		 */
		MICROUI_IMAGE_FORMAT_A2,

		/**
		 * Defines an image whose pixel format is Alpha4.
		 */
		MICROUI_IMAGE_FORMAT_A4,

		/**
		 * Defines an image whose pixel format is Alpha8.
		 */
		MICROUI_IMAGE_FORMAT_A8,

		/**
		 * Defines an image whose pixel format is Color1.
		 */
		MICROUI_IMAGE_FORMAT_C1,

		/**
		 * Defines an image whose pixel format is Color2.
		 */
		MICROUI_IMAGE_FORMAT_C2,

		/**
		 * Defines an image whose pixel format is Color4.
		 */
		MICROUI_IMAGE_FORMAT_C4,

		/**
		 * Defines an image whose pixel format is Alpha1-Color1.
		 */
		MICROUI_IMAGE_FORMAT_AC11,

		/**
		 * Defines an image whose pixel format is Alpha2-Color2.
		 */
		MICROUI_IMAGE_FORMAT_AC22,

		/**
		 * Defines an image whose pixel format is Alpha4-Color4.
		 */
		MICROUI_IMAGE_FORMAT_AC44,

		/**
		 * Defines an image whose pixel format is a LUT entry on 8 bits and target an ARGB8888 color.
		 */
		MICROUI_IMAGE_FORMAT_LARGB8888,

		/**
		 * Defines an image whose pixel format is a LUT entry on 8 bits and target a RGB888 color.
		 */
		MICROUI_IMAGE_FORMAT_LRGB888,

		/**
		 * Defines an image whose pixel format is MicroUI implementation dependent.
		 */
		MICROUI_IMAGE_FORMAT_INTERNAL,
	}

	/**
	 * Represents a MicroUI Image.
	 * <p>
	 * This interface is used by several drawing methods which use an image as source image. It can be mapped on byte
	 * array given as parameter in some MicroUI natives. This byte array is retrieved in MicroEJ application using the
	 * method Image.getSNIContext().
	 *
	 * @see LLUIDisplay#mapMicroUIImage(byte[])
	 * @see LLUIDisplay#newMicroUIImage(byte[])
	 */
	public interface MicroUIImage {

		/**
		 * Returns the MicroUI Image width.
		 *
		 * @return image width in pixels.
		 */
		int getWidth();

		/**
		 * Returns the MicroUI Image height.
		 *
		 * @return image height in pixels.
		 */
		int getHeight();

		/**
		 * Returns the MicroUI Image format.
		 *
		 * @return image format.
		 */
		MicroUIImageFormat getFormat();

		/**
		 * Returns the front panel image allocated for the MicroUI Image.
		 *
		 * @return the MicroUI Image representation in front panel framework.
		 */
		Image getImage();

		/**
		 * Returns the front panel image allocated for the MicroUI Image, applying a rendering color.
		 *
		 * This color is only useful when the MicroUI Image format is
		 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_A8}, {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_A4},
		 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_A2} or {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_A1}.
		 *
		 * Next calls to {@link #getImage()} (in same native function) will return the same image with same rendering
		 * color until next call to {@link #getImage(int)}.
		 *
		 * For all other formats, the color is ignored and this call is equivalent than calling {@link #getImage()}.
		 *
		 * @param color
		 *            the color to apply on Ax image in 0xRRGGBB format (the alpha component is ignored).
		 *
		 * @return the MicroUI Image representation in front panel framework.
		 */
		Image getImage(int color);

		/**
		 * Returns true when the image targets the LCD.
		 *
		 * @return true when the image targets the LCD.
		 */
		boolean isLCD();

		/**
		 * Returns true when the image has been closed by the MicroEJ application.
		 *
		 * @return true when the image has been closed.
		 */
		boolean isClosed();
	}

	/**
	 * Represents a MicroUI GraphicsContext.
	 * <p>
	 * This interface is used by all drawing functions to target the destination. It can be mapped on byte array given
	 * as parameter in MicroUI natives. This byte array is retrieved in MicroEJ application using the method
	 * GraphicsContext.getSNIContext().
	 *
	 * @see LLUIDisplay#mapMicroUIGraphicsContext(byte[])
	 * @see LLUIDisplay#newMicroUIGraphicsContext(byte[])
	 */
	public interface MicroUIGraphicsContext extends MicroUIImage {

		/**
		 * Returns the current MicroUI drawing color to use to perform drawings.
		 *
		 * @return a color in 0xAARRGGBB format (where alpha level is always 0xff == fully opaque).
		 */
		int getMicroUIColor();

		/**
		 * Returns the color to use to render the drawings. This color takes in consideration the embedded display pixel
		 * representation constraints.
		 * <p>
		 * Equivalent of <code>LLUIDisplay.Instance.convertARGBColorToColorToDraw(getMicroUIColor());</code>
		 *
		 * @return the color to use to render the drawings in 0xAARRGGBB format (where alpha level is always 0xff ==
		 *         fully opaque).
		 * @see LLUIDisplay#convertARGBColorToColorToDraw(int)
		 */
		int getRenderingColor();

		/**
		 * @return top-left X coordinate of current clip.
		 */
		int getClipX1();

		/**
		 * @return top-left Y coordinate of current clip.
		 */
		int getClipY1();

		/**
		 * @return bottom-right X coordinate of current clip.
		 */
		int getClipX2();

		/**
		 * @return bottom-right Y coordinate of current clip.
		 */
		int getClipY2();

		/**
		 * Enables or disables the graphics context clip. Useful to ignore clip during a drawing. The clip is
		 * automatically re-enabled when calling requestDrawing().
		 *
		 * @param enable
		 *            false to disable the clip.
		 */
		void configureClip(boolean enable);

		/**
		 * Tells if the clip is enabled or not. This call should be performed at the beginning of a drawing function in
		 * order to prevent to make some useless operations when the clip is disabled.
		 * <p>
		 * When it is disabled, that means the caller considers the drawing is fitting the clip. In this case checking
		 * the clip bounds is useless.
		 *
		 * @return true when the clip is enabled (the clip must be checked).
		 */
		boolean isClipEnabled();

		/**
		 * Tells if given point fits the clip or not.
		 *
		 * @param x
		 *            the pixel X coordinate.
		 * @param y
		 *            the pixel Y coordinate.
		 *
		 * @return false when the point is outside the clip.
		 */
		boolean isPointInClip(int x, int y);

		/**
		 * Tells if given horizontal line fully fits the clip or not.
		 *
		 * @param x1
		 *            the first pixel line X coordinate.
		 * @param x2
		 *            the last pixel line X coordinate.
		 * @param y
		 *            the both pixels line Y coordinate.
		 *
		 * @return false when the line is fully or partially outside the clip, true when line fully fits the clip (clip
		 *         can be disabled).
		 */
		boolean isHorizontalLineInClip(int x1, int x2, int y);

		/**
		 * Tells if given vertical line fully fits the clip or not.
		 *
		 * @param x
		 *            the both pixels line X coordinate.
		 * @param y1
		 *            the first pixel line Y coordinate.
		 * @param y2
		 *            the last pixel line Y coordinate.
		 *
		 * @return false when the line is fully or partially outside the clip, true when line fully fits the clip (clip
		 *         can be disabled).
		 */
		boolean isVerticalLineInClip(int y1, int y2, int x);

		/**
		 * Tells if given rectangle fully fits the clip or not.
		 *
		 * @param x1
		 *            the top-left pixel X coordinate.
		 * @param y1
		 *            the top-left pixel Y coordinate.
		 * @param x2
		 *            the bottom-right pixel X coordinate.
		 * @param y2
		 *            the top-right pixel Y coordinate.
		 *
		 * @return false when the rectangle is fully or partially outside the clip, true when rectangle fully fits the
		 *         clip (clip can be disabled).
		 */
		boolean isRectangleInClip(int x1, int y1, int x2, int y2);

		/**
		 * Tells if given region (from x,y to x+width-1,y+height-1) fully fits the clip or not.
		 *
		 * @param x
		 *            the top-left pixel X coordinate.
		 * @param y
		 *            the top-left pixel Y coordinate.
		 * @param width
		 *            the region width.
		 * @param height
		 *            the region height.
		 *
		 * @return false when the region is fully or partially outside the clip, true when region fully fits the clip
		 *         (clip can be disabled).
		 */
		boolean isRegionInClip(int x, int y, int width, int height);

		/**
		 * Tells if source and destination share a region.
		 *
		 * Source and destination can share a region if and only if source and destination target the same MicroUI
		 * Image. In that case, this function checks if there is an intersection of source and destination regions. This
		 * function is useful when drawing an image on same image.
		 *
		 * @param gc
		 *            the MicroUI GraphicsContext target of destination.
		 * @param img
		 *            the MicroUI Image to draw.
		 * @param regionX
		 *            the x coordinate of the upper-left corner of the region to check.
		 * @param regionY
		 *            the x coordinate of the upper-left corner of the region to check.
		 * @param width
		 *            the width of the region to check.
		 * @param height
		 *            the height of the region to check.
		 * @param destX
		 *            the x coordinate of the top-left point in the destination.
		 * @param destY
		 *            the y coordinate of the top-left point in the destination.
		 *
		 * @return true when source and destination are same image and when destination region intersects source region.
		 */
		boolean regionsOverlap(MicroUIImage img, int regionX, int regionY, int width, int height, int destX, int destY);

		/**
		 * Tells if the ellipsis is enabled or not. Returns 0 when ellipsis is disabled, a positive value otherwise.
		 * This value is the maximum string width in pixels. If string width is higher than this value, an ellipsis will
		 * be drawn to crop the string.
		 *
		 * @return the ellipsis width or 0.
		 */
		int getEllipsisWidth();

		/**
		 * Requests a drawing on the graphics context.
		 *
		 * @return true if drawing can be performed, false when the drawing does not need to be done for any reason.
		 */
		boolean requestDrawing();

		/**
		 * Updates the flush dirty area. This dirty area will be merged with the current dirty area and given as
		 * parameter in {@link LLUIDisplayImpl#flush(MicroUIGraphicsContext, Image, int, int, int, int)} method.
		 * <p>
		 * This method is useless when the given graphics context is not the display (in this case this call has no
		 * effect). Caller should check that calling {@link #isLCD()} before.
		 *
		 * @param xmin
		 *            the dirty area top-left X coordinate.
		 * @param ymin
		 *            the dirty area top-left Y coordinate.
		 * @param xmax
		 *            the dirty area bottom-right X coordinate.
		 * @param ymax
		 *            the dirty area bottom-right Y coordinate.
		 * @return true when dirty area has been updated or false when the specified dirty area is out of current clip
		 *         or when the given graphics context does not target the display.
		 */
		boolean setDrawingLimits(int xmin, int ymin, int xmax, int ymax);
	}

	/**
	 * Current drawer used by the implementation of MicroUI drawings native methods.
	 *
	 * @see #setDrawer(UIDrawing)
	 * @see #getDrawer()
	 */
	private static UIDrawing Drawer;

	private LLUIPainter() {
		// forbidden constructor: there is no instance of LLUIPainter
	}

	/**
	 * Sets the given drawer as current drawer. This drawer is used by all implementation of MicroUI drawings native
	 * methods.
	 * <p>
	 * The drawer has the responsibility to check the clip (draw outside current clip is forbidden) and has to update
	 * the new <code>Display.flush()</code> area (dirty area) as mentioned in {@link LLUIPainter} class comment.
	 * <p>
	 * Default drawer is the MicroUI graphical engine's internal software drawer.
	 *
	 * @param drawer
	 *            the new drawer to use or null to use the MicroUI graphical engine's internal software drawer.
	 */
	public static void setDrawer(UIDrawing drawer) {
		Drawer = drawer == null ? LLUIDisplay.Instance.getUIDrawerSoftware() : drawer;
	}

	/**
	 * Returns the current drawer used by all implementation of MicroUI drawings native methods.
	 *
	 * @return the current drawer (MicroUI graphical engine's internal software drawer if no other drawer has been set).
	 */
	public static UIDrawing getDrawer() {
		return Drawer;
	}

	/**
	 * MicroUI native "writePixel". Draws a pixel at given position.
	 * <p>
	 * See class comment to have more information about native method implementation.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param x
	 *            the pixel X coordinate.
	 * @param y
	 *            the pixel Y coordinate.
	 */
	public static void writePixel(byte[] target, int x, int y) {
		LLUIDisplay graphicalEngine = LLUIDisplay.Instance;
		synchronized (graphicalEngine) {
			MicroUIGraphicsContext gc = graphicalEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && gc.isPointInClip(x, y)) {
				gc.configureClip(false /* point is in clip */);
				getDrawer().writePixel(gc, x, y);
			}
		}
	}

	/**
	 * MicroUI native "drawLine". Draws a line at between points x1,y1 (included) and x2,y2 (included). Note: x1 may be
	 * higher than x2 (and/or y1 may be higher than y2).
	 * <p>
	 * See class comment to have more information about native method implementation.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param x1
	 *            the first pixel line X coordinate.
	 * @param y1
	 *            the first pixel line Y coordinate.
	 * @param x2
	 *            the last pixel line X coordinate.
	 * @param y2
	 *            the last pixel line Y coordinate.
	 */
	public static void drawLine(byte[] target, int x1, int y1, int x2, int y2) {
		LLUIDisplay graphicalEngine = LLUIDisplay.Instance;
		synchronized (graphicalEngine) {
			MicroUIGraphicsContext gc = graphicalEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing()) {
				// cannot reduce/clip line: may be x2 < x1 and / or y2 < y1
				getDrawer().drawLine(gc, x1, y1, x2, y2);
			}
		}
	}

	/**
	 * MicroUI native "drawHorizontalLine". Draws a horizontal line at between points x,y (included) and x+width-1,y
	 * (included).
	 * <p>
	 * See class comment to have more information about native method implementation.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param x
	 *            the first pixel line X coordinate.
	 * @param y
	 *            the first pixel line Y coordinate.
	 * @param width
	 *            the line size.
	 */
	public static void drawHorizontalLine(byte[] target, int x, int y, int width) {
		LLUIDisplay graphicalEngine = LLUIDisplay.Instance;
		synchronized (graphicalEngine) {
			MicroUIGraphicsContext gc = graphicalEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && width > 0) {
				int x1 = x;
				int x2 = x + width - 1;
				// can disable clip when line is fully in clip
				gc.configureClip(!gc.isHorizontalLineInClip(x1, x2, y));
				getDrawer().drawHorizontalLine(gc, x1, x2, y);
			}
		}
	}

	/**
	 * MicroUI native "drawVerticalLine". Draws a vertical line at between points x,y (included) and x,y+height-1
	 * (included).
	 * <p>
	 * See class comment to have more information about native method implementation.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param x
	 *            the first pixel line X coordinate.
	 * @param y
	 *            the first pixel line Y coordinate.
	 * @param height
	 *            the line size.
	 */
	public static void drawVerticalLine(byte[] target, int x, int y, int height) {
		LLUIDisplay graphicalEngine = LLUIDisplay.Instance;
		synchronized (graphicalEngine) {
			MicroUIGraphicsContext gc = graphicalEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && height > 0) {
				int y1 = y;
				int y2 = y + height - 1;
				// can disable clip when line is fully in clip
				gc.configureClip(!gc.isVerticalLineInClip(y1, y2, x));
				getDrawer().drawVerticalLine(gc, x, y1, y2);
			}
		}
	}

	/**
	 * MicroUI native "drawRectangle". Draws a rectangle at from top-left point x,y (included) and bottom-right point
	 * x+width-1,y+height-1 (included).
	 * <p>
	 * See class comment to have more information about native method implementation.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the rectangle width.
	 * @param height
	 *            the rectangle height.
	 */
	public static void drawRectangle(byte[] target, int x, int y, int width, int height) {
		LLUIDisplay graphicalEngine = LLUIDisplay.Instance;
		synchronized (graphicalEngine) {
			MicroUIGraphicsContext gc = graphicalEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && width > 0 && height > 0) {
				int x1 = x;
				int x2 = x + width - 1;
				int y1 = y;
				int y2 = y + height - 1;
				// can disable clip when rectangle is fully in clip
				gc.configureClip(!gc.isRectangleInClip(x1, y1, x2, y2));
				getDrawer().drawRectangle(gc, x1, y1, x2, y2);
			}
		}
	}

	/**
	 * MicroUI native "fillRectangle". Fills a rectangle at from top-left point x1,y1 (included) and bottom-right point
	 * x2,y2 (included). Caller ensures that x1 <= x2 and y1 <= y2.
	 * <p>
	 * See class comment to have more information about native method implementation.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the rectangle width.
	 * @param height
	 *            the rectangle height.
	 */
	public static void fillRectangle(byte[] target, int x, int y, int width, int height) {
		LLUIDisplay graphicalEngine = LLUIDisplay.Instance;
		synchronized (graphicalEngine) {
			MicroUIGraphicsContext gc = graphicalEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && width > 0 && height > 0) {
				int x1 = x;
				int x2 = x + width - 1;
				int y1 = y;
				int y2 = y + height - 1;
				// can disable clip when rectangle is fully in clip
				gc.configureClip(!gc.isRectangleInClip(x1, y1, x2, y2));
				getDrawer().fillRectangle(gc, x1, y1, x2, y2);
			}
		}
	}

	/**
	 * MicroUI native "drawRoundedRectangle". Draws a rounded rectangle at from top-left point x,y (included) and
	 * bottom-right point x+width-1,y+height-1 (included).
	 *
	 * @param target
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the rectangle width.
	 * @param height
	 *            the rectangle height.
	 * @param arcWidth
	 *            the horizontal diameter of the arc at the corners.
	 * @param arcHeight
	 *            the vertical diameter of the arc at the corners.
	 */
	public static void drawRoundedRectangle(byte[] target, int x, int y, int width, int height, int arcWidth,
			int arcHeight) {
		LLUIDisplay graphicalEngine = LLUIDisplay.Instance;
		synchronized (graphicalEngine) {
			MicroUIGraphicsContext gc = graphicalEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && width > 0 && height > 0) {
				// can disable clip when rectangle is fully in clip
				gc.configureClip(!gc.isRegionInClip(x, y, width, height));
				getDrawer().drawRoundedRectangle(gc, x, y, width, height, arcWidth, arcHeight);
			}
		}
	}

	/**
	 * MicroUI native "fillRoundedRectangle". Fills a rounded rectangle at from top-left point x,y (included) and
	 * bottom-right point x+width-1,y+height-1 (included).
	 *
	 * @param target
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the rectangle width.
	 * @param height
	 *            the rectangle height.
	 * @param arcWidth
	 *            the horizontal diameter of the arc at the corners.
	 * @param arcHeight
	 *            the vertical diameter of the arc at the corners.
	 */
	public static void fillRoundedRectangle(byte[] target, int x, int y, int width, int height, int arcWidth,
			int arcHeight) {
		LLUIDisplay graphicalEngine = LLUIDisplay.Instance;
		synchronized (graphicalEngine) {
			MicroUIGraphicsContext gc = graphicalEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && width > 0 && height > 0) {
				// can disable clip when rectangle is fully in clip
				gc.configureClip(!gc.isRegionInClip(x, y, width, height));
				getDrawer().fillRoundedRectangle(gc, x, y, width, height, arcWidth, arcHeight);
			}
		}
	}

	/**
	 * MicroUI native "drawCircleArc". Draws a circular arc covering the square defined by top-left point x,y (included)
	 * and bottom-right point x+diameter-1,y+diameter-1 (included).
	 * <p>
	 * The arc is drawn from startAngle up to arcAngle degrees. The center of the arc is defined as the center of the
	 * square whose origin is at (x,y) (upper-left corner) and whose dimension is given by diameter.
	 * <p>
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 * <p>
	 * If either the given diameter is negative or zero, or if arcAngle is zero, nothing is drawn.
	 * <p>
	 * The angles are given relative to the square. For instance an angle of 45 degrees is always defined by the line
	 * from the center of the square to the upper right corner of the square.
	 *
	 * @param target
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param diameter
	 *            the diameter of the arc to draw.
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from startAngle
	 */
	public static void drawCircleArc(byte[] target, int x, int y, int diameter, float startAngle, float arcAngle) {
		LLUIDisplay graphicalEngine = LLUIDisplay.Instance;
		synchronized (graphicalEngine) {
			MicroUIGraphicsContext gc = graphicalEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && diameter > 0 && arcAngle != 0) {
				// can disable clip when arc is fully in clip
				gc.configureClip(!gc.isRegionInClip(x, y, diameter, diameter));
				getDrawer().drawCircleArc(gc, x, y, diameter, startAngle, arcAngle);
			}
		}
	}

	/**
	 * MicroUI native "drawEllipseArc". Draws an elliptical arc covering the rectangle defined by top-left point x,y
	 * (included) and bottom-right point x+width-1,y+height-1 (included).
	 * <p>
	 * The arc is drawn from startAngle up to arcAngle degrees. The center of the arc is defined as the center of the
	 * rectangle whose origin is at (x,y) (upper-left corner) and whose dimension is given by width and height.
	 * <p>
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 * <p>
	 * If either the given width or height is negative or zero, or if arcAngle is zero, nothing is drawn.
	 * <p>
	 * The angles are given relative to the rectangle. For instance an angle of 45 degrees is always defined by the line
	 * from the center of the rectangle to the upper right corner of the rectangle. Thus for a non squared rectangle
	 * angles are skewed along either height or width.
	 *
	 * @param target
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the rectangle width.
	 * @param height
	 *            the rectangle height.
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from startAngle
	 */
	public static void drawEllipseArc(byte[] target, int x, int y, int width, int height, float startAngle,
			float arcAngle) {
		LLUIDisplay graphicalEngine = LLUIDisplay.Instance;
		synchronized (graphicalEngine) {
			MicroUIGraphicsContext gc = graphicalEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && width > 0 && height > 0 && arcAngle != 0) {
				// can disable clip when arc is fully in clip
				gc.configureClip(!gc.isRegionInClip(x, y, width, height));
				getDrawer().drawEllipseArc(gc, x, y, width, height, startAngle, arcAngle);
			}
		}
	}

	/**
	 * MicroUI native "fillCircleArc". Fills a circular arc covering the square defined by top-left point x,y (included)
	 * and bottom-right point x+diameter-1,y+diameter-1 (included).
	 *
	 * The arc is drawn from startAngle up to arcAngle degrees. The center of the arc is defined as the center of the
	 * square whose origin is at (x,y) (upper-left corner) and whose dimension is given by width and height.
	 *
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 *
	 * If either the given diameter is negative or zero, or if arcAngle is zero, nothing is drawn.
	 *
	 * The angles are given relative to the square. For instance an angle of 45 degrees is always defined by the line
	 * from the center of the square to the upper right corner of the square.
	 *
	 * @param target
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param diameter
	 *            the diameter of the arc to draw.
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from startAngle
	 */
	public static void fillCircleArc(byte[] target, int x, int y, int diameter, float startAngle, float arcAngle) {
		LLUIDisplay graphicalEngine = LLUIDisplay.Instance;
		synchronized (graphicalEngine) {
			MicroUIGraphicsContext gc = graphicalEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && diameter > 0 && arcAngle != 0) {
				// can disable clip when arc is fully in clip
				gc.configureClip(!gc.isRegionInClip(x, y, diameter, diameter));
				getDrawer().fillCircleArc(gc, x, y, diameter, startAngle, arcAngle);
			}
		}
	}

	/**
	 * MicroUI native "fillEllipseArc". Fills an arc covering the rectangle defined by top-left point x,y (included) and
	 * bottom-right point x+width-1,y+height-1 (included).
	 * <p>
	 * The arc is drawn from startAngle up to arcAngle degrees. The center of the arc is defined as the center of the
	 * rectangle whose origin is at (x,y) (upper-left corner) and whose dimension is given by width and height.
	 * <p>
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 * <p>
	 * If either the given width or height is negative or zero, or if arcAngle is zero, nothing is drawn.
	 * <p>
	 * The angles are given relative to the rectangle. For instance an angle of 45 degrees is always defined by the line
	 * from the center of the rectangle to the upper right corner of the rectangle. Thus for a non squared rectangle
	 * angles are skewed along either height or width.
	 *
	 * @param target
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the rectangle width.
	 * @param height
	 *            the rectangle height.
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from startAngle
	 */
	public static void fillEllipseArc(byte[] target, int x, int y, int width, int height, float startAngle,
			float arcAngle) {
		LLUIDisplay graphicalEngine = LLUIDisplay.Instance;
		synchronized (graphicalEngine) {
			MicroUIGraphicsContext gc = graphicalEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && width > 0 && height > 0 && arcAngle != 0) {
				// can disable clip when arc is fully in clip
				gc.configureClip(!gc.isRegionInClip(x, y, width, height));
				getDrawer().fillEllipseArc(gc, x, y, width, height, startAngle, arcAngle);
			}
		}
	}

	/**
	 * MicroUI native "drawEllipse". Draws an ellipse covering the rectangle defined by top-left point x,y (included)
	 * and bottom-right point x+width-1,y+height-1 (included).
	 * <p>
	 * If either the given width or height is negative or zero, nothing is drawn.
	 *
	 * @param target
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the ellipse width.
	 * @param height
	 *            the ellipse height.
	 */

	public static void drawEllipse(byte[] target, int x, int y, int width, int height) {
		LLUIDisplay graphicalEngine = LLUIDisplay.Instance;
		synchronized (graphicalEngine) {
			MicroUIGraphicsContext gc = graphicalEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && width > 0 && height > 0) {
				// can disable clip when ellipse is fully in clip
				gc.configureClip(!gc.isRegionInClip(x, y, width, height));
				getDrawer().drawEllipse(gc, x, y, width, height);

			}
		}
	}

	/**
	 * MicroUI native "fillEllipse". Fills an ellipse covering the rectangle defined by top-left point x,y (included)
	 * and bottom-right point x+width-1,y+height-1 (included).
	 * <p>
	 * If either the given width or height is negative or zero, nothing is drawn.
	 *
	 * @param target
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the ellipse width.
	 * @param height
	 *            the ellipse height.
	 */
	public static void fillEllipse(byte[] target, int x, int y, int width, int height) {
		LLUIDisplay graphicalEngine = LLUIDisplay.Instance;
		synchronized (graphicalEngine) {
			MicroUIGraphicsContext gc = graphicalEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && width > 0 && height > 0) {
				// can disable clip when ellipse is fully in clip
				gc.configureClip(!gc.isRegionInClip(x, y, width, height));
				getDrawer().fillEllipse(gc, x, y, width, height);
			}
		}
	}

	/**
	 * MicroUI native "drawCircle". Draws a circle covering the square defined by top-left point x,y (included) and
	 * bottom-right point x+diameter-1,y+diameter-1 (included).
	 * <p>
	 * If the given diameter is negative or zero, nothing is drawn.
	 *
	 * @param target
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param diameter
	 *            the circle square size.
	 */
	public static void drawCircle(byte[] target, int x, int y, int diameter) {
		LLUIDisplay graphicalEngine = LLUIDisplay.Instance;
		synchronized (graphicalEngine) {
			MicroUIGraphicsContext gc = graphicalEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && diameter > 0) {
				// can disable clip when circle is fully in clip
				gc.configureClip(!gc.isRegionInClip(x, y, diameter, diameter));
				getDrawer().drawCircle(gc, x, y, diameter);
			}
		}
	}

	/**
	 * MicroUI native "fillCircle". Fills a circle covering the square defined by top-left point x,y (included) and
	 * bottom-right point x+diameter-1,y+diameter-1 (included).
	 * <p>
	 * If the given diameter is negative or zero, nothing is drawn.
	 *
	 * @param target
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param diameter
	 *            the circle square size.
	 */
	public static void fillCircle(byte[] target, int x, int y, int diameter) {
		LLUIDisplay graphicalEngine = LLUIDisplay.Instance;
		synchronized (graphicalEngine) {
			MicroUIGraphicsContext gc = graphicalEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && diameter > 0) {
				// can disable clip when circle is fully in clip
				gc.configureClip(!gc.isRegionInClip(x, y, diameter, diameter));
				getDrawer().fillCircle(gc, x, y, diameter);
			}
		}
	}

	/**
	 * MicroUI native "drawImage". Draws a region of an image. The region of the image to draw is given relative to the
	 * image (origin at the upper-left corner) as a rectangle.
	 * <p>
	 * If the specified source region exceeds the image bounds, the copied region is limited to the image boundary. If
	 * the copied region goes out of the bounds of the graphics context area, pixels out of the range will not be drawn.
	 * <p>
	 * A global opacity value is given. When this value is 0xff (255, opaque), that means the image is drawn on the
	 * graphics context without managing an extra opacity. Only the image transparent pixels must have been merged with
	 * destination. All image opaque pixels override destination.
	 * <p>
	 * When this value is a value between 0 and 0xff, that means each pixel of the image must be merged with destination
	 * in addition with the image transparent pixels. An image opaque pixel becomes transparent (its opacity is the
	 * given alpha) and the opacity of an image transparent pixel becomes (alpha * alpha(pixel)) / 255.
	 *
	 *
	 * @param target
	 *            the MicroUI GraphicsContext target.
	 * @param source
	 *            the MicroUI Image to draw.
	 * @param regionX
	 *            the x coordinate of the upper-left corner of the region to copy.
	 * @param regionY
	 *            the x coordinate of the upper-left corner of the region to copy.
	 * @param width
	 *            the width of the region to copy.
	 * @param height
	 *            the height of the region to copy.
	 * @param x
	 *            the x coordinate of the top-left point in the destination.
	 * @param y
	 *            the y coordinate of the top-left point in the destination.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	public static void drawImage(byte[] target, byte[] source, int regionX, int regionY, int width, int height, int x,
			int y, int alpha) {
		LLUIDisplay graphicalEngine = LLUIDisplay.Instance;
		synchronized (graphicalEngine) {
			MicroUIGraphicsContext gc = graphicalEngine.mapMicroUIGraphicsContext(target);
			MicroUIImage image = graphicalEngine.mapMicroUIImage(source);
			if (gc.requestDrawing() && !image.isClosed() && alpha > 0) {
				// compute inside image bounds
				if (regionX < 0) {
					width += regionX; // decrease size
					x -= regionX; // increase origin
					regionX = 0;
				}
				if (regionX + width > image.getWidth()) {
					width = image.getWidth() - regionX; // decrease size
				}
				if (regionY < 0) {
					height += regionY; // decrease size
					y -= regionY; // increase origin
					regionY = 0;
				}
				if (regionY + height > image.getHeight()) {
					height = image.getHeight() - regionY; // decrease size
				}

				// compute inside graphics context bounds
				if (x < 0) {
					width += x; // decrease size
					regionX -= x; // increase origin
					x = 0;
				}
				if (x + width > gc.getWidth()) {
					width = gc.getWidth() - x; // decrease size
				}
				if (y < 0) {
					height += y; // decrease size
					regionY -= y; // increase origin
					y = 0;
				}
				if (y + height > gc.getHeight()) {
					height = gc.getHeight() - y; // decrease size
				}

				if (width > 0 && height > 0) {
					// can disable clip when region is fully in clip
					gc.configureClip(!gc.isRegionInClip(x, y, width, height));
					getDrawer().drawImage(gc, image, regionX, regionY, width, height, x, y, alpha);
				}
				// else: nothing to do
			}
			// else: refused drawing
		}
	}

}
