/*
 * Copyright 2019-2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */

package ej.net.util.ssl;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.cert.Certificate;
import java.security.cert.CertificateFactory;
import java.util.Objects;

import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;

/**
 * Helps to create an SSL context.
 */
public class SslContextBuilder {

	/**
	 * TLS version 1.2.
	 */
	public static final String TLS_VERSION_1_2 = "TLSv1.2";

	/**
	 * TLS version 1.3.
	 */
	public static final String TLS_VERSION_1_3 = "TLSv1.3";

	private static final String CERT_TYPE = "X509";
	private static final String KEYSTORE_TYPE = "PKCS12";

	private final String protocol;
	private final KeyStore trustStore;
	private final KeyStore keyStore;
	private int aliasId;

	/**
	 * Creates an SSL context builder.
	 *
	 * @throws GeneralSecurityException
	 *             if an exception occurs while initializing the trust store or the key store.
	 */
	public SslContextBuilder() throws GeneralSecurityException {
		this(TLS_VERSION_1_2); //
	}

	/**
	 * Creates an SSL context builder for the given TLS protocol version.
	 *
	 * @param protocol
	 *            the TLS protocol version string.
	 * @throws GeneralSecurityException
	 *             if an exception occurs while initializing the trust store or the key store.
	 * @see SSLContext#getInstance(String)
	 */
	public SslContextBuilder(String protocol) throws GeneralSecurityException {
		this.trustStore = createTrustStoreInstance();
		this.keyStore = createKeyStoreInstance();
		this.aliasId = 0;
		this.protocol = protocol;
	}

	/**
	 * Adds the given server certificate to the trust store.
	 *
	 * @param certificate
	 *            the certificate to add.
	 * @throws GeneralSecurityException
	 *             if the certificate could not be loaded or added.
	 */
	public void addServerCertificate(InputStream certificate) throws GeneralSecurityException {
		Certificate cert = generateCertificate(certificate);
		this.trustStore.setCertificateEntry(getNewAlias(), cert);
	}

	/**
	 * Adds the given server certificate to the trust store.
	 *
	 * Tries to find the resource in the application's classpath (using {@link Class#getResourceAsStream(String)}).
	 *
	 * @param certificatePath
	 *            the path to the certificate to add.
	 * @throws GeneralSecurityException
	 *             if the certificate could not be loaded or added.
	 * @throws IOException
	 *             if an I/O occurs when loading the resource.
	 */
	public void addServerCertificate(String certificatePath) throws GeneralSecurityException, IOException {
		try (InputStream certificate = loadResource(certificatePath)) {
			addServerCertificate(certificate);
		}
	}

	/**
	 * Adds the given client key and the associated certificate and certification chain to the key store.
	 *
	 * @param privateKey
	 *            the private key to add.
	 * @param certificate
	 *            the associated certificate.
	 * @param certificationChain
	 *            the certification chain, ordered, the root certificate at the end.
	 * @throws GeneralSecurityException
	 *             if the certificates could not be loaded or the private key could not be added.
	 */
	public void addClientKey(byte[] privateKey, InputStream certificate, InputStream... certificationChain)
			throws GeneralSecurityException {
		Certificate cert = generateCertificate(certificate);
		Certificate[] certChain = new Certificate[1 + certificationChain.length];
		certChain[0] = cert;
		for (int i = 0; i < certificationChain.length; i++) {
			InputStream currentCert = certificationChain[i];
			Objects.requireNonNull(currentCert);
			certChain[i + 1] = generateCertificate(currentCert);
		}
		this.keyStore.setKeyEntry(getNewAlias(), privateKey, certChain);
	}

	/**
	 * Adds the given client key and the associated self-signed certificate.
	 *
	 * @param privateKey
	 *            the private key to add.
	 * @param certificate
	 *            the associated certificate.
	 * @throws GeneralSecurityException
	 *             if the certificates could not be loaded or the private key could not be added.
	 */
	public void addClientKey(byte[] privateKey, InputStream certificate) throws GeneralSecurityException {
		addClientKey(privateKey, certificate, new InputStream[0]);
	}

	/**
	 * Adds the given client key and the associated certificate and certification chain to the key store.
	 *
	 * @param privateKey
	 *            the private key to add.
	 * @param certificate
	 *            the associated certificate.
	 * @param certificationChain
	 *            the certification chain, ordered, the root certificate at the end.
	 * @throws GeneralSecurityException
	 *             if the certificate could not be loaded or the private key could not be added.
	 * @throws IOException
	 *             if the given privateKey could not be read.
	 */
	public void addClientKey(InputStream privateKey, InputStream certificate, InputStream... certificationChain)
			throws IOException, GeneralSecurityException {
		addClientKey(readInputStream(privateKey), certificate, certificationChain);
	}

	/**
	 * Adds the given client key and the associated self-signed certificate.
	 *
	 * @param privateKey
	 *            the private key to add.
	 * @param certificate
	 *            the associated certificate.
	 * @throws GeneralSecurityException
	 *             if the certificate could not be loaded or the private key could not be added.
	 * @throws IOException
	 *             if the given privateKey could not be read.
	 */
	public void addClientKey(InputStream privateKey, InputStream certificate)
			throws IOException, GeneralSecurityException {
		addClientKey(readInputStream(privateKey), certificate, new InputStream[0]);
	}

	/**
	 * Adds the given client key and the associated certificate and certification chain to the key store.
	 *
	 * Tries to find the resource in the application's classpath (using {@link Class#getResourceAsStream(String)}).
	 *
	 * @param privateKeyPath
	 *            the path to the private key to add.
	 * @param certificatePath
	 *            the path to the associated certificate.
	 * @param certificationChainPaths
	 *            the paths to the certification chain, ordered, the root certificate at the end.
	 * @throws GeneralSecurityException
	 *             if the certificate could not be loaded or the private key could not be added.
	 * @throws IOException
	 *             if the given privateKey could not be read.
	 */
	public void addClientKey(String privateKeyPath, String certificatePath, String... certificationChainPaths)
			throws IOException, GeneralSecurityException {
		InputStream[] certificationChain = null;
		try (InputStream privateKey = loadResource(privateKeyPath);
				InputStream certificate = loadResource(certificatePath)) {
			certificationChain = new InputStream[certificationChainPaths.length];
			for (int i = 0; i < certificationChainPaths.length; i++) {
				String currentCertificationChainPath = certificationChainPaths[i];
				Objects.requireNonNull(currentCertificationChainPath);
				certificationChain[i] = loadResource(currentCertificationChainPath);
			}
			addClientKey(privateKey, certificate, certificationChain);
		} finally {
			if (certificationChain != null) {
				for (int i = 0; i < certificationChain.length; i++) {
					if (certificationChain[i] != null) {
						certificationChain[i].close();
					}
				}
			}
		}
	}

	/**
	 * Creates an SSL context using the populated trust store and an empty key store.
	 *
	 * @return the created SSL context.
	 * @throws GeneralSecurityException
	 *             if an exception occurred while creating the SSL context.
	 */
	public SSLContext build() throws GeneralSecurityException {
		// create trust store
		TrustManagerFactory tm = TrustManagerFactory.getInstance(CERT_TYPE);
		tm.init(this.trustStore);
		TrustManager[] trustManagers = tm.getTrustManagers();

		// create SSL context
		SSLContext sslContext = SSLContext.getInstance(this.protocol);
		sslContext.init(null, trustManagers, null);
		return sslContext;
	}

	/**
	 * Creates an SSL context using the populated trust store and an the populated key store, using the given key store
	 * password.
	 *
	 * @return the created SSL context.
	 * @param keyStorePassword
	 *            the password of the key store.
	 * @throws GeneralSecurityException
	 *             if an exception occurred while creating the SSL context.
	 */
	public SSLContext build(String keyStorePassword) throws GeneralSecurityException {
		// create trust store
		TrustManagerFactory tm = TrustManagerFactory.getInstance(CERT_TYPE);
		tm.init(this.trustStore);
		TrustManager[] trustManagers = tm.getTrustManagers();

		// create key store
		KeyManagerFactory km = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
		km.init(this.keyStore, keyStorePassword.toCharArray());
		KeyManager[] keyManagers = km.getKeyManagers();

		// create SSL context
		SSLContext sslContext = SSLContext.getInstance(this.protocol);
		sslContext.init(keyManagers, trustManagers, null);
		return sslContext;
	}

	private String getNewAlias() {
		return Integer.toString(this.aliasId++);
	}

	private static KeyStore createTrustStoreInstance() throws GeneralSecurityException {
		KeyStore trustStore = KeyStore.getInstance(KeyStore.getDefaultType());
		try {
			trustStore.load(null, null);
		} catch (IOException e) {
			throw new KeyStoreException(e);
		}
		return trustStore;
	}

	private static KeyStore createKeyStoreInstance() throws GeneralSecurityException {
		KeyStore keyStore;
		try {
			keyStore = KeyStore.getInstance(KEYSTORE_TYPE);
		} catch (KeyStoreException e) { // NOSONAR - this exception is expected
			keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
		}
		try {
			keyStore.load(null, null);
		} catch (IOException e) {
			throw new KeyStoreException(e);
		}
		return keyStore;
	}

	private static Certificate generateCertificate(InputStream inputStream) throws GeneralSecurityException {
		CertificateFactory certificateFactory = CertificateFactory.getInstance(CERT_TYPE);
		return certificateFactory.generateCertificate(inputStream);
	}

	private static byte[] readInputStream(InputStream inputStream) throws IOException {
		byte[] data = new byte[1024];
		ByteArrayOutputStream out = new ByteArrayOutputStream(1024);

		int bytesRead;
		while ((bytesRead = inputStream.read(data)) > 0) {
			out.write(data, 0, bytesRead);
		}

		return out.toByteArray();
	}

	private InputStream loadResource(String path) {
		InputStream is = getClass().getResourceAsStream(path);
		if (is == null) {
			throw new IllegalArgumentException("Cannot load " + path); //$NON-NLS-1$
		}
		return is;
	}

}
