/*
 * Java
 *
 * Copyright 2016-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.hoka.http;

import ej.hoka.http.requesthandler.RequestHandler;

/**
 * Represent an HTTP route.
 */
public class Route {

	static final String DEFAULT_ACCEPT_TYPE = "*/*"; //$NON-NLS-1$

	private final int httpMethod;
	private final String path;
	private final String acceptType;
	private final RequestHandler handler;

	/**
	 *
	 * @param httpMethod
	 *            HTTP method {@link HttpRequest#GET} {@link HttpRequest#POST} {@link HttpRequest#DELETE}
	 *            {@link HttpRequest#PUT}
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public Route(final int httpMethod, String path, final RequestHandler handler) {
		this(httpMethod, path, null, handler);
	}

	/**
	 *
	 * @param httpMethod
	 *            HTTP method {@link HttpRequest#GET} {@link HttpRequest#POST} {@link HttpRequest#DELETE}
	 *            {@link HttpRequest#PUT}
	 * @param path
	 *            request path
	 *
	 * @param acceptType
	 *            accepted content type
	 * @param handler
	 *            request handler
	 */
	public Route(final int httpMethod, String path, final String acceptType, final RequestHandler handler) {
		this.handler = handler;
		this.httpMethod = httpMethod;
		this.acceptType = acceptType == null ? null : acceptType.trim().toLowerCase();
		this.path = path;
	}

	/**
	 *
	 * @return the path (URI) mapped in this route.
	 */
	public String getPath() {
		return this.path;
	}

	/**
	 *
	 * @return http method. The mapping between the method number and the name can be found in
	 *
	 *         {@link HttpRequest#GET} {@link HttpRequest#POST} {@link HttpRequest#PUT} {@link HttpRequest#DELETE}
	 *
	 *         {@link Route#getHttpMethodAsString()} can be used to get the string representation of the http method.
	 */
	public int getHttpMethod() {
		return this.httpMethod;
	}

	/**
	 * Gets the acceptType.
	 *
	 * @return the acceptType.
	 */
	public String getAcceptType() {
		return this.acceptType;
	}

	/**
	 *
	 * @return return the request handler associated with this route.
	 */
	public RequestHandler getHandler() {
		return this.handler;
	}

	/**
	 *
	 * @return true is this route accept all content type
	 */
	public boolean acceptAllContentTypes() {
		return this.acceptType == null || DEFAULT_ACCEPT_TYPE.equals(this.acceptType);
	}

	/**
	 *
	 * @return http method as string
	 */
	public String getHttpMethodAsString() {
		switch (this.httpMethod) {
		case HttpRequest.BEFORE_ALL:
			return "BEFORE_ALL"; //$NON-NLS-1$ ;
		case HttpRequest.BEFORE:
			return "BEFORE"; //$NON-NLS-1$ ;
		case HttpRequest.AFTER:
			return "AFTER"; //$NON-NLS-1$ ;
		case HttpRequest.AFTER_ALL:
			return "AFTER_ALL"; //$NON-NLS-1$ ;
		case HttpRequest.GET:
			return HttpConstants.HTTP_METHOD_GET;
		case HttpRequest.POST:
			return HttpConstants.HTTP_METHOD_POST;
		case HttpRequest.PUT:
			return HttpConstants.HTTP_METHOD_PUT;
		case HttpRequest.DELETE:
			return HttpConstants.HTTP_METHOD_DELETE;
		case HttpRequest.HEAD:
			return HttpConstants.HTTP_METHOD_HEAD;
		case HttpRequest.CONNECT:
			return HttpConstants.HTTP_METHOD_CONNECT;
		case HttpRequest.OPTIONS:
			return HttpConstants.HTTP_METHOD_OPTIONS;
		case HttpRequest.TRACE:
			return HttpConstants.HTTP_METHOD_TRACE;
		case HttpRequest.PATCH:
			return HttpConstants.HTTP_METHOD_PATCH;

		default:
			throw new IllegalArgumentException("Http method not supported"); //$NON-NLS-1$
		}
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((this.acceptType == null) ? 0 : this.acceptType.hashCode());
		result = prime * result + this.httpMethod;
		result = prime * result + ((this.path == null) ? 0 : this.path.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		Route other = (Route) obj;
		if (this.acceptType == null) {
			if (other.acceptType != null) {
				return false;
			}
		} else if (!this.acceptType.equals(other.acceptType)) {
			return false;
		}
		if (this.httpMethod != other.httpMethod) {
			return false;
		}
		if (this.path == null) {
			if (other.path != null) {
				return false;
			}
		} else if (!this.path.equals(other.path)) {
			return false;
		}
		return true;
	}

}
