/*
 * Copyright (c) 1997, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2015-2020 MicroEJ Corp. EDC compliance and optimizations.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package java.security;

import java.security.spec.AlgorithmParameterSpec;

import ej.annotation.Nullable;

/**
 * This class defines the <i>Service Provider Interface</i> (<b>SPI</b>) for the {@code Signature} class, which is used
 * to provide the functionality of a digital signature algorithm. Digital signatures are used for authentication and
 * integrity assurance of digital data. .
 * <p>
 * All the abstract methods in this class must be implemented by each cryptographic service provider who wishes to
 * supply the implementation of a particular signature algorithm.
 *
 * @author Benjamin Renaud
 *
 *
 * @see Signature
 */

public abstract class SignatureSpi {

	/**
	 * Application-specified source of randomness.
	 */
	@Nullable
	protected SecureRandom appRandom = null;

	/**
	 * Initializes this signature object with the specified public key for verification operations.
	 *
	 * @param publicKey
	 *            the public key of the identity whose signature is going to be verified.
	 *
	 * @exception InvalidKeyException
	 *                if the key is improperly encoded, parameters are missing, and so on.
	 */
	protected abstract void engineInitVerify(PublicKey publicKey) throws InvalidKeyException;

	/**
	 * Initializes this signature object with the specified private key for signing operations.
	 *
	 * @param privateKey
	 *            the private key of the identity whose signature will be generated.
	 *
	 * @exception InvalidKeyException
	 *                if the key is improperly encoded, parameters are missing, and so on.
	 */
	protected abstract void engineInitSign(PrivateKey privateKey) throws InvalidKeyException;

	/**
	 * Initializes this signature object with the specified private key and source of randomness for signing operations.
	 *
	 * <p>
	 * This concrete method has been added to this previously-defined abstract class. (For backwards compatibility, it
	 * cannot be abstract.)
	 *
	 * @param privateKey
	 *            the private key of the identity whose signature will be generated.
	 * @param random
	 *            the source of randomness
	 *
	 * @exception InvalidKeyException
	 *                if the key is improperly encoded, parameters are missing, and so on.
	 */
	protected void engineInitSign(PrivateKey privateKey, SecureRandom random) throws InvalidKeyException {
		throw new RuntimeException();
	}

	/**
	 * Updates the data to be signed or verified using the specified byte.
	 *
	 * @param b
	 *            the byte to use for the update.
	 *
	 * @exception SignatureException
	 *                if the engine is not initialized properly.
	 */
	protected abstract void engineUpdate(byte b) throws SignatureException;

	/**
	 * Updates the data to be signed or verified, using the specified array of bytes, starting at the specified offset.
	 *
	 * @param b
	 *            the array of bytes
	 * @param off
	 *            the offset to start from in the array of bytes
	 * @param len
	 *            the number of bytes to use, starting at offset
	 *
	 * @exception SignatureException
	 *                if the engine is not initialized properly
	 */
	protected abstract void engineUpdate(byte[] b, int off, int len) throws SignatureException;

	/**
	 * Returns the signature bytes of all the data updated so far. The format of the signature depends on the underlying
	 * signature scheme.
	 *
	 * @return the signature bytes of the signing operation's result.
	 *
	 * @exception SignatureException
	 *                if the engine is not initialized properly or if this signature algorithm is unable to process the
	 *                input data provided.
	 */
	protected abstract byte[] engineSign() throws SignatureException;

	/**
	 * Finishes this signature operation and stores the resulting signature bytes in the provided buffer {@code outbuf},
	 * starting at {@code offset}. The format of the signature depends on the underlying signature scheme.
	 *
	 * <p>
	 * The signature implementation is reset to its initial state (the state it was in after a call to one of the
	 * {@code engineInitSign} methods) and can be reused to generate further signatures with the same private key.
	 *
	 * This method should be abstract, but we leave it concrete for binary compatibility. Knowledgeable providers should
	 * override this method.
	 *
	 * @param outbuf
	 *            buffer for the signature result.
	 *
	 * @param offset
	 *            offset into {@code outbuf} where the signature is stored.
	 *
	 * @param len
	 *            number of bytes within {@code outbuf} allotted for the signature. Both this default implementation and
	 *            the SUN provider do not return partial digests. If the value of this parameter is less than the actual
	 *            signature length, this method will throw a SignatureException. This parameter is ignored if its value
	 *            is greater than or equal to the actual signature length.
	 *
	 * @return the number of bytes placed into {@code outbuf}
	 *
	 * @exception SignatureException
	 *                if the engine is not initialized properly, if this signature algorithm is unable to process the
	 *                input data provided, or if {@code len} is less than the actual signature length.
	 *
	 * @since 1.2
	 */
	protected int engineSign(byte[] outbuf, int offset, int len) throws SignatureException {
		throw new RuntimeException();
	}

	/**
	 * Verifies the passed-in signature.
	 *
	 * @param sigBytes
	 *            the signature bytes to be verified.
	 *
	 * @return true if the signature was verified, false if not.
	 *
	 * @exception SignatureException
	 *                if the engine is not initialized properly, the passed-in signature is improperly encoded or of the
	 *                wrong type, if this signature algorithm is unable to process the input data provided, etc.
	 */
	protected abstract boolean engineVerify(byte[] sigBytes) throws SignatureException;

	/**
	 * Verifies the passed-in signature in the specified array of bytes, starting at the specified offset.
	 *
	 * <p>
	 * Note: Subclasses should overwrite the default implementation.
	 *
	 *
	 * @param sigBytes
	 *            the signature bytes to be verified.
	 * @param offset
	 *            the offset to start from in the array of bytes.
	 * @param length
	 *            the number of bytes to use, starting at offset.
	 *
	 * @return true if the signature was verified, false if not.
	 *
	 * @exception SignatureException
	 *                if the engine is not initialized properly, the passed-in signature is improperly encoded or of the
	 *                wrong type, if this signature algorithm is unable to process the input data provided, etc.
	 * @since 1.4
	 */
	protected boolean engineVerify(byte[] sigBytes, int offset, int length) throws SignatureException {
		throw new RuntimeException();
	}

	/**
	 * <p>
	 * This method is overridden by providers to initialize this signature engine with the specified parameter set.
	 *
	 * @param params
	 *            the parameters
	 *
	 * @exception UnsupportedOperationException
	 *                if this method is not overridden by a provider
	 *
	 * @exception InvalidAlgorithmParameterException
	 *                if this method is overridden by a provider and the given parameters are inappropriate for this
	 *                signature engine
	 */
	protected void engineSetParameter(AlgorithmParameterSpec params) throws InvalidAlgorithmParameterException {
		throw new UnsupportedOperationException();
	}

	/**
	 * Returns a clone if the implementation is cloneable.
	 *
	 * @return a clone if the implementation is cloneable.
	 *
	 * @exception CloneNotSupportedException
	 *                if this is called on an implementation that does not support {@code Cloneable}.
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		if (this instanceof Cloneable) {
			return super.clone();
		} else {
			throw new CloneNotSupportedException();
		}
	}
}
