/*
	Copyright (c) 2006 JSON.org
	Copyright 2016 IS2T. This file has been modified by IS2T S.A.

	Permission is hereby granted, free of charge, to any person obtaining a copy
	of this software and associated documentation files (the "Software"), to deal
	in the Software without restriction, including without limitation the rights
	to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
	copies of the Software, and to permit persons to whom the Software is
	furnished to do so, subject to the following conditions:

	The above copyright notice and this permission notice shall be included in all
	copies or substantial portions of the Software.

	The Software shall be used for Good, not Evil.

	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
	FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
	AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
	LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
	OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
	SOFTWARE.
 */
package ej.json;

import java.io.IOException;
import java.io.OutputStream;


/**
 * JSONWriter provides a quick and convenient way of producing JSON text.
 * The texts produced strictly conform to JSON syntax rules. No whitespace is
 * added, so the results are ready for transmission or storage. Each instance of
 * JSONWriter can produce one JSON text.
 * <p>
 * A JSONWriter instance provides a <code>value</code> method for appending
 * values to the
 * text, and a <code>key</code>
 * method for adding keys before values in objects. There are <code>array</code>
 * and <code>endArray</code> methods that make and bound array values, and
 * <code>object</code> and <code>endObject</code> methods which make and bound
 * object values. All of these methods return the JSONWriter instance,
 * permitting a cascade style. For example, <pre>
 * new JSONWriter(myOutputStream)
 *     .object()
 *         .key("JSON")
 *         .value("Hello, World!")
 *     .endObject();</pre> which writes <pre>
 * {"JSON":"Hello, World!"}</pre>
 * <p>
 * The first method called must be <code>array</code> or <code>object</code>.
 * There are no methods for adding commas or colons. JSONWriter adds them for
 * you. Objects and arrays can be nested up to 20 levels deep.
 * <p>
 * @author JSON.org
 * @version 2
 */
public class JSONWriter {
	private static final int maxdepth = 20;

	private static final int ERROR_NULL_POINTER 			= 0;
	private static final int ERROR_VALUE_OUT_OF_SEQUENCE 	= 1;
	private static final int ERROR_MISPLACED_ARRAY 			= 2;
	private static final int ERROR_MISPLACED_ENDOBJECT		= 3;
	private static final int ERROR_MISPLACED_ENDARRAY		= 4;
	private static final int ERROR_NULL_KEY					= 5;
	private static final int ERROR_MISPLACED_KEY			= 6;
	private static final int ERROR_MISPLACED_OBJECT			= 7;
	private static final int ERROR_NESTING_ERROR			= 8;
	private static final int ERROR_NESTING_TOO_DEEP			= 9;
	private static final int ERROR_NON_FINITE_NUMBERS		= 10;

	/**
	 * The comma flag determines if a comma should be output before the next
	 * value.
	 */
	private boolean comma;

	/**
	 * The current mode. Values:
	 * 'a' (array),
	 * 'd' (done),
	 * 'i' (initial),
	 * 'k' (key),
	 * 'o' (object).
	 */
	protected char mode;

	/**
	 * The object/array stack.
	 */
	private final char stack[];

	/**
	 * The stack top index. A value of 0 indicates that the stack is empty.
	 */
	private int top;

	/**
	 * The stream that will receive the output.
	 */
	protected OutputStream os;

	/**
	 * Make a fresh JSONWriter. It can be used to build one JSON text.
	 * @param w the output stream
	 */
	public JSONWriter(OutputStream os) {
		this.comma = false;
		this.mode = 'i';
		this.stack = new char[maxdepth];
		this.top = 0;
		this.os = os;
	}

	/**
	 * Append a value.
	 * @param s A string value.
	 * @return this
	 * @throws JSONException If the value is out of sequence.
	 */
	private JSONWriter append(String s) throws JSONException {
		if (s == null) {
			throw error(ERROR_NULL_POINTER);
		}
		if (this.mode == 'o' || this.mode == 'a' || this.mode == 'i') {
			try {
				if (this.comma && this.mode == 'a') {
					this.os.write(',');
				}
				this.os.write(s.getBytes("UTF-8"));
			}
			catch (IOException e) {
				throw new JSONException(e);
			}
			if (this.mode == 'o') {
				this.mode = 'k';
			}
			else if(this.mode == 'i') {
				this.mode = 'd';
			}
			this.comma = true;
			return this;
		}
		throw error(ERROR_VALUE_OUT_OF_SEQUENCE);
	}

	/**
	 * Begin appending a new array. All values until the balancing
	 * <code>endArray</code> will be appended to this array. The
	 * <code>endArray</code> method must be called to mark the array's end.
	 * @return this
	 * @throws JSONException If the nesting is too deep, or if the object is
	 * started in the wrong place (for example as a key or after the end of the
	 * outermost array or object).
	 */
	public JSONWriter array() throws JSONException {
		if (this.mode == 'i' || this.mode == 'o' || this.mode == 'a') {
			this.push('a');
			this.append("[");
			this.comma = false;
			return this;
		}
		throw error(ERROR_MISPLACED_ARRAY);
	}

	/**
	 * End something.
	 * @param m Mode
	 * @param c Closing character
	 * @return this
	 * @throws JSONException If unbalanced.
	 */
	private JSONWriter end(char m, char c) throws JSONException {
		if (this.mode != m) {
			throw error(m == 'o' ? ERROR_MISPLACED_ENDOBJECT : ERROR_MISPLACED_ENDARRAY);
		}
		this.pop(m);
		try {
			this.os.write(c);
		} catch (IOException e) {
			throw new JSONException(e);
		}
		this.comma = true;
		return this;
	}

	/**
	 * End an array. This method most be called to balance calls to
	 * <code>array</code>.
	 * @return this
	 * @throws JSONException If incorrectly nested.
	 */
	public JSONWriter endArray() throws JSONException {
		return this.end('a', ']');
	}

	/**
	 * End an object. This method most be called to balance calls to
	 * <code>object</code>.
	 * @return this
	 * @throws JSONException If incorrectly nested.
	 */
	public JSONWriter endObject() throws JSONException {
		return this.end('k', '}');
	}

	/**
	 * Append a key. The key will be associated with the next value. In an
	 * object, every value must be preceded by a key.
	 * @param s A key string.
	 * @return this
	 * @throws JSONException If the key is out of place. For example, keys
	 *  do not belong in arrays or if the key is null.
	 */
	public JSONWriter key(String s) throws JSONException {
		if (s == null) {
			throw error(ERROR_NULL_KEY);
		}
		if (this.mode == 'k') {
			try {
				if (this.comma) {
					this.os.write(',');
				}
				this.os.write(quote(s).getBytes("UTF-8"));
				this.os.write(':');
				this.comma = false;
				this.mode = 'o';
				return this;
			} catch (IOException e) {
				throw new JSONException(e);
			}
		}
		throw error(ERROR_MISPLACED_KEY);
	}


	/**
	 * Begin appending a new object. All keys and values until the balancing
	 * <code>endObject</code> will be appended to this object. The
	 * <code>endObject</code> method must be called to mark the object's end.
	 * @return this
	 * @throws JSONException If the nesting is too deep, or if the object is
	 * started in the wrong place (for example as a key or after the end of the
	 * outermost array or object).
	 */
	public JSONWriter object() throws JSONException {
		if (this.mode == 'i') {
			this.mode = 'o';
		}
		if (this.mode == 'o' || this.mode == 'a') {
			this.append("{");
			this.push('k');
			this.comma = false;
			return this;
		}
		throw error(ERROR_MISPLACED_OBJECT);

	}


	/**
	 * Pop an array or object scope.
	 * @param c The scope to close.
	 * @throws JSONException If nesting is wrong.
	 */
	private void pop(char c) throws JSONException {
		if (this.top <= 0 || this.stack[this.top - 1] != c) {
			throw error(ERROR_NESTING_ERROR);
		}
		this.top -= 1;
		this.mode = this.top == 0 ? 'd' : this.stack[this.top - 1];
	}

	/**
	 * Push an array or object scope.
	 * @param c The scope to open.
	 * @throws JSONException If nesting is too deep.
	 */
	private void push(char c) throws JSONException {
		if (this.top >= maxdepth) {
			throw error(ERROR_NESTING_TOO_DEEP);
		}
		this.stack[this.top] = c;
		this.mode = c;
		this.top += 1;
	}


	/**
	 * Append either the value <code>true</code> or the value
	 * <code>false</code>.
	 * @param b A boolean.
	 * @return this
	 * @throws JSONException
	 */
	public JSONWriter value(boolean b) throws JSONException {
		return this.append(b ? "true" : "false");
	}

	/**
	 * Append a double value.
	 * @param d A double.
	 * @return this
	 * @throws JSONException If the number is not finite.
	 */
	public JSONWriter value(double d) throws JSONException {
		if (((Double)d).isInfinite() || ((Double)d).isNaN()) {
			throw error(ERROR_NON_FINITE_NUMBERS);
		}
		return this.append(trimNumber(Double.toString(d)));
	}

	/**
	 * Shave off trailing zeros and decimal point, if possible.
	 */
	private static String trimNumber(String s) {
		if (s.indexOf('.') > 0 && s.indexOf('e') < 0 && s.indexOf('E') < 0) {
			while (s.endsWith("0")) {
				s = s.substring(0, s.length() - 1);
			}
			if (s.endsWith(".")) {
				s = s.substring(0, s.length() - 1);
			}
		}
		return s;
	}

	/**
	 * Append a long value.
	 * @param l A long.
	 * @return this
	 * @throws JSONException
	 */
	public JSONWriter value(long l) throws JSONException {
		return this.append(Long.toString(l));
	}


	/**
	 * Append a String value.
	 * @param s A String
	 * @return this
	 * @throws JSONException
	 */
	public JSONWriter value(String s) throws JSONException {
		return this.append(quote(s));
	}

	/**
	 * Produce a string in double quotes with backslash sequences in all the right places. A backslash will be inserted
	 * within &lt;/, allowing JSON text to be delivered in HTML. In JSON text, a string cannot contain a control
	 * character or an unescaped quote or backslash.
	 *
	 * @param string
	 *            A String
	 * @return A String correctly formatted for insertion in a JSON text.
	 */
	public static String quote(String string) {
		if (string == null || string.length() == 0) {
			return "\"\"";
		}

		char b;
		char c = 0;
		int i;
		int len = string.length();
		StringBuffer sb = new StringBuffer(len + 4);
		String t;

		sb.append('"');
		for (i = 0; i < len; i += 1) {
			b = c;
			c = string.charAt(i);
			switch (c) {
			case '\\':
			case '"':
				sb.append('\\');
				sb.append(c);
				break;
			case '/':
				if (b == '<') {
					sb.append('\\');
				}
				sb.append(c);
				break;
			case '\b':
				sb.append("\\b");
				break;
			case '\t':
				sb.append("\\t");
				break;
			case '\n':
				sb.append("\\n");
				break;
			case '\f':
				sb.append("\\f");
				break;
			case '\r':
				sb.append("\\r");
				break;
			default:
				if (c < ' ') {
					t = "000" + Integer.toHexString(c);
					sb.append("\\u" + t.substring(t.length() - 4));
				} else {
					sb.append(c);
				}
			}
		}
		sb.append('"');
		return sb.toString();
	}

	/**
	 * Make a JSONException to signal a writing error.
	 *
	 * @param code The error code.
	 * @return  A JSONException object, suitable for throwing
	 */
	private JSONException error(int code) {
		return new JSONException("E="+code);
	}
}
