# Overview

Class Extender is a tool that allows to extend a class by adding methods from another class. An extension
can be either a static or an instance method.

An extension method is a static method annotated with `@Extend` annotation. This annotation takes two arguments:
- `className` that takes a `String` representing the fully qualified name of the class to which this method must be
added.
- `isStatic` (optional), that takes a boolean value telling that this method must remain static or not. When `false`
(default value), the `static` modifier is removed, it thus defines an instance method.

An instance extension method must have a first argument of the type of the class to extend. This argument represents
the reference to the instance (the `this`). It will be removed when the extension is applied. The name of this argument is
not important. Note that the extension method must have a `static` modifier which will also be removed when injected in
the target class.

You are free to choose the name of the class that holds the extension methods.

This tool works at binary level. It means that extensions can be provided as binary artifacts and applied to binary
libraries without access to source code.

## Limitations

- Extensions only have access to public fields and methods of the extended class.
- Extension methods must be self-contained. They cannot be split into several methods (unless these methods already
exist in the class to extend): the method it depends on will not be injected into the targeted class.
- An extension cannot redefine a method of the target class (same name and arguments). Nevertheless, methods defined
in super classes can be overridden.

# Build

Class Extender build will produce an executable fat jar `class-extender.jar` that contains all its dependencies.

See [MicroEJ documentation](https://docs.microej.com/en/latest/SDKUserGuide/mmm.html?highlight=build#module-build) for
more information on how to build a MicroEJ Module.

# Usage

## Extend a class

Given a Java class `example.Simple` with a `hello()` method like the following:

```Java
package example;

public class Simple {

	public String hello() {
		return "hello";
	}
}
```

We will extend it by adding a new method `helloExtended()`.

Note:
>The `@Extended` annotation is provided by the `basictool` library version >= `1.7.0` that needs to be added to the extension project by adding the following dependency `<dependency org="ej.library.runtime" name="basictool" rev="1.7.0"/>`


For that, let's create a new class `example.SimpleExtension` as follows

```Java
package example;

import ej.basictool.annotation.Extend;


public class SimpleExtension {

	@Extend(className = "example.Simple")
	public static String helloExtended(Simple self) {
		return self.hello() + " extended";
	}
}
```

The `Extend` annotation indicates that the `helloExtended()` method should be added to the `example.Simple` class. 
The first parameter of the extension method must be of the type of the class to extend. It will be used as the reference to the extended class by the Class Extender Tool.

This class can be compiled and then passed as an argument to the tool along with the classpath containing the `example.Simple`
class file. The tool generates a new copy of `example.Simple` class file that contains the `helloExtended` method.
The extended class generated will be equivalent to

```Java
package example;

public class Simple {

	public String hello() {
		return "hello";
	}
	
	public String helloExtended() {
		return this.hello() + " extended";
	}
}
```

Extensions classes can be packed in MicroEJ modules (using `microej-javalib` build type for instance) to build
artifacts that can be used as dependencies to other extension projects.

## Use Class Extender as an Executable JAR

The `class-extender.jar` file is an executable JAR file that contains all its dependencies. It can be called in
command line with

```
java -jar class-extender.jar <arguments>
```

where `<arguments>` are
- the classpath in which extensions will be searched. It is a list of directories and JAR files separated with a
system-dependant classpath separator: `;` on Windows, and `:` on Unix-like systems. This classpath must include all
the libraries needed for extensions (i.e.: you may need to include EDC).
- the path of the JAR file to extend
- (OPTIONAL) the path of the output directory in which the extended JAR file will be copied. When not set, the original jar designated by the second argument will be modified.

**WARNING** `class-extender.jar` tool does not verify arguments correctness. It supposes that you pass correct
arguments in the right order.

## Use Class Extender as an Ant Task

Class Extender also defines an [Ant task](src/main/java/com/microej/tool/classextender/ClassExtenderTask.java). This
task takes two or three parameters:
- `classpath`: the classpath in which extensions will be searched
- `sourceJar`: the path of the JAR file to extend
- `targetDirectory`: the path of the output directory in which the extended JAR file will be copied. This parameter is
optional


See the [Use Class Extender as an Executable JAR](#use-class-extender-as-an-ant-task) section above for more
information about arguments.

```
<taskdef name="ClassExtender" classname="com.microej.tool.classextender.ClassExtenderTask" classpathref="classextender.classpath.ref"/>	
```

where `classextender.classpath.ref` is the path id of the class-extender tool. For instance:

```XML
<path id="classextender.classpath.ref">
	<fileset dir="<class-extender tool dir>" includes="*.jar" />
</path>
```
where `<class-extender tool dir>` is the directory containing `class-extender.jar`.

# Include Class Extender During Firmware Project Build 

A `module.ant` script can be added to your firmware configuration project in order to extend libraries during build:


```XML
<project name="module" xmlns:ea="antlib:org.apache.easyant" xmlns:ivy="antlib:org.apache.ivy.ant">

	<dirname property="project.dir" file="${ant.file.module}"/>
	<property name="build.target.dir" value="${project.dir}/target~"/>
	<property name="dependencies.dir" value="${build.target.dir}/dependencies"/>
	<property name="target.platform.dir" value="${build.target.dir}/virtualDevice"/>

	<target name="module-extend:extend" extensionOf="abstract-compile:compile-ready">
		<!-- ClassExtender tool definition -->
		<ivy:cachepath pathid="build.classpath" conf="build" />
		<taskdef name="ClassExtender" classname="com.microej.tool.classextender.ClassExtenderTask" classpathref="build.classpath"/>

		<!--
			Define class extensions classpath.
			All JAR files or classes directories found in "extensions" directory will be considered.
		-->
		<path id="classextension.classpath.ref">
			<fileset dir="${dependencies.dir}" includes="*.jar"/>
		</path>
		<pathconvert property="classextension.classpath" refid="classextension.classpath.ref"/>

		<!-- Define paths to EDC implementations (emb and sim). -->
		<path id="edc.emb.ref">
			<fileset dir="${target.platform.dir}/MICROJVM/javaLibs" includes="edc-*.jar" />
		</path>
		<pathconvert property="edc.emb.path" refid="edc.emb.ref"/>

		<path id="edc.sim.ref">
			<fileset dir="${target.platform.dir}/S3/javaLibs" includes="edc-*.jar" />
		</path>
		<pathconvert property="edc.sim.path" refid="edc.sim.ref"/>

		<!-- Save S3 EDC in to new directory -->
		<property name="modified.jar.directory" value="${target.platform.dir}/S3/javaLibs/original"/>
		<mkdir dir="${modified.jar.directory}"/>
		<copy file="${edc.sim.path}" todir="${modified.jar.directory}"/>

		<!-- Applies extensions to embedded EDC-->
		<ClassExtender classpath="${classextension.classpath}" sourceJar="${edc.emb.path}"/>

		<!-- Applies extensions to sim EDC-->
		<ClassExtender classpath="${classextension.classpath}" sourceJar="${edc.sim.path}"/>

		<!-- Use the original EDC in boot classpath (used by debugger) -->
		<replace file="${target.platform.dir}/scripts/init-edc-s3/edcinit.xml" token="${s3.libs.dir}" value="${s3.libs.dir}/original"/>
	</target>
</project>
```

The provided script only extends [EDC](https://docs.microej.com/en/latest/overview/runtime.html?highlight=edc#embedded-device-configuration-edc)
for now.

- Copy and paste the contents of the above ANT script in a `module.ant` file at the root of your firmware project.
- Add extensions as dependencies in `module.ivy` file.
- Add `<conf name="build" visibility="private" description="Build time dependencies."/>` between `<configuration></configuration>` tags in `module.ivy`.
- Add `<dependency org="com.microej.tool" name="class-extender" rev="1.0.4" conf="build->default" transitive="false"/>` as new dependency in `module.ivy`.

# Requirements

- Java SE 8

# Dependencies

_All dependencies are retrieved transitively by MicroEJ Module Manager_.

# Source

N/A.

# Restrictions

None.

---  
_Copyright 2022 MicroEJ Corp. All rights reserved._  
_Use of this source code is governed by a BSD-style license that can be found with this software._  
