/*
 * Java
 *
 * Copyright 2021-2022 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.hoka.log;

import ej.hoka.http.Config;

/**
 * Simple logger implementation print messages to System#out
 */
class SimpleLogger implements Logger {

	private static final int NONE = 0x00000;
	private static final int TRACE = 0x11111;
	private static final int DEBUG = 0x01111;
	private static final int INFO = 0x00111;
	private static final int WARN = 0x00011;
	private static final int ERROR = 0x00001;

	private static final String TRACE_PREFIX = "[TRACE]\t"; //$NON-NLS-1$
	private static final String DEBUG_PREFIX = "[DEBUG]\t"; //$NON-NLS-1$
	private static final String INFO_PREFIX = "[INFO]\t"; //$NON-NLS-1$
	private static final String WARN_PREFIX = "[WARN]\t"; //$NON-NLS-1$
	private static final String ERROR_PREFIX = "[ERROR]\t"; //$NON-NLS-1$

	private final int level;

	SimpleLogger() {
		String slevel = Config.getInstance().getLoggerLevel();
		if (slevel.trim().isEmpty()) {
			this.level = INFO; // use info for default log level
		} else {
			switch (slevel.trim().toLowerCase()) {
			case "none": //$NON-NLS-1$
				this.level = NONE;
				break;
			case "trace": //$NON-NLS-1$
				this.level = TRACE;
				break;
			case "debug": //$NON-NLS-1$
				this.level = DEBUG;
				break;
			case "info": //$NON-NLS-1$
				this.level = INFO;
				break;
			case "warn": //$NON-NLS-1$
				this.level = WARN;
				break;
			case "error": //$NON-NLS-1$
				this.level = ERROR;
				break;
			default:
				this.level = INFO;
				break;
			}
		}
	}

	@Override
	public void trace(final String msg) {
		if (isTraceEnabled()) {
			System.out.println(TRACE_PREFIX + msg); // NOSONAR
		}
	}

	@Override
	public void debug(final String msg) {
		if (isDebugEnabled()) {
			System.out.println(DEBUG_PREFIX + msg); // NOSONAR
		}
	}

	@Override
	public void info(final String msg) {
		if (isInfoEnabled()) {
			System.out.println(INFO_PREFIX + msg); // NOSONAR
		}
	}

	@Override
	public void warn(final String msg) {
		if (isWarnEnabled()) {
			System.out.println(WARN_PREFIX + msg); // NOSONAR
		}
	}

	@Override
	public void error(final String msg) {
		if (isErrorEnabled()) {
			System.out.println(ERROR_PREFIX + msg); // NOSONAR
		}
	}

	@Override
	public void error(final String msg, final Throwable e) {
		if (isErrorEnabled()) {
			System.out.println(ERROR_PREFIX + msg); // NOSONAR
			e.printStackTrace(); // NOSONAR
		}

	}

	@Override
	public void error(final Throwable e) {
		if (isErrorEnabled()) {
			System.out.println(ERROR_PREFIX + e.getMessage()); // NOSONAR
			e.printStackTrace(); // NOSONAR
		}
	}

	@Override
	public boolean isTraceEnabled() {
		return (TRACE & this.level) == TRACE;
	}

	@Override
	public boolean isDebugEnabled() {
		return (DEBUG & this.level) == DEBUG;
	}

	@Override
	public boolean isInfoEnabled() {
		return (INFO & this.level) == INFO;
	}

	@Override
	public boolean isWarnEnabled() {
		return (WARN & this.level) == WARN;
	}

	@Override
	public boolean isErrorEnabled() {
		return (ERROR & this.level) == ERROR;
	}

}
