/*
 * Java
 *
 * Copyright 2016-2022 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.hoka.http;

import ej.annotation.Nullable;
import ej.hoka.http.requesthandler.RequestHandler;

/**
 * Represent an HTTP route.
 */
public class Route {

	static final String DEFAULT_ACCEPT_TYPE = "*/*"; //$NON-NLS-1$

	private final int httpMethod;
	private final String path;
	@Nullable
	private final String acceptType;
	private final RequestHandler handler;

	/**
	 * Constructs a new instance of Route.
	 *
	 * @param httpMethod
	 *            HTTP method {@link HttpRequest#GET} {@link HttpRequest#POST} {@link HttpRequest#DELETE}
	 *            {@link HttpRequest#PUT}
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public Route(final int httpMethod, String path, final RequestHandler handler) {
		this(httpMethod, path, null, handler);
	}

	/**
	 * Constructs a new instance of Route.
	 *
	 * @param httpMethod
	 *            HTTP method {@link HttpRequest#GET} {@link HttpRequest#POST} {@link HttpRequest#DELETE}
	 *            {@link HttpRequest#PUT}
	 * @param path
	 *            request path
	 *
	 * @param acceptType
	 *            accepted content type
	 * @param handler
	 *            request handler
	 */
	public Route(final int httpMethod, String path, @Nullable final String acceptType, final RequestHandler handler) {
		this.handler = handler;
		this.httpMethod = httpMethod;
		this.acceptType = acceptType == null ? null : acceptType.trim().toLowerCase();
		this.path = path;
	}

	/**
	 * Gets the path (URI) mapped in this route.
	 *
	 * @return the path (URI) mapped in this route.
	 */
	public String getPath() {
		return this.path;
	}

	/**
	 * Gets the integer value associated with the HTTP method of the route.
	 *
	 * @return http method. The mapping between the method number and the name can be found in
	 *
	 *         {@link HttpRequest#GET} {@link HttpRequest#POST} {@link HttpRequest#PUT} {@link HttpRequest#DELETE}
	 *
	 *         {@link Route#getHttpMethodAsString()} can be used to get the string representation of the http method.
	 */
	public int getHttpMethod() {
		return this.httpMethod;
	}

	/**
	 * Gets the acceptType.
	 *
	 * @return the acceptType.
	 */
	@Nullable
	public String getAcceptType() {
		return this.acceptType;
	}

	/**
	 * Gets the request handler associated with this route.
	 *
	 * @return the request handler associated with this route.
	 */
	public RequestHandler getHandler() {
		return this.handler;
	}

	/**
	 * Checks if this route accepts all content types.
	 *
	 * @return true if this route accepts all content types.
	 */
	public boolean acceptAllContentTypes() {
		return this.acceptType == null || DEFAULT_ACCEPT_TYPE.equals(this.acceptType);
	}

	/**
	 * Returns the HTTP method of this route as a String.
	 *
	 * @return http method as string
	 */
	public String getHttpMethodAsString() {
		switch (this.httpMethod) {
		case HttpRequest.BEFORE_ALL:
			return "BEFORE_ALL"; //$NON-NLS-1$
		case HttpRequest.BEFORE:
			return "BEFORE"; //$NON-NLS-1$
		case HttpRequest.AFTER:
			return "AFTER"; //$NON-NLS-1$
		case HttpRequest.AFTER_ALL:
			return "AFTER_ALL"; //$NON-NLS-1$
		case HttpRequest.GET:
			return HttpConstants.HTTP_METHOD_GET;
		case HttpRequest.POST:
			return HttpConstants.HTTP_METHOD_POST;
		case HttpRequest.PUT:
			return HttpConstants.HTTP_METHOD_PUT;
		case HttpRequest.DELETE:
			return HttpConstants.HTTP_METHOD_DELETE;
		case HttpRequest.HEAD:
			return HttpConstants.HTTP_METHOD_HEAD;
		case HttpRequest.CONNECT:
			return HttpConstants.HTTP_METHOD_CONNECT;
		case HttpRequest.OPTIONS:
			return HttpConstants.HTTP_METHOD_OPTIONS;
		case HttpRequest.TRACE:
			return HttpConstants.HTTP_METHOD_TRACE;
		case HttpRequest.PATCH:
			return HttpConstants.HTTP_METHOD_PATCH;

		default:
			throw new IllegalArgumentException("Http method not supported"); //$NON-NLS-1$
		}
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		String acceptType = this.acceptType;
		result = prime * result + ((acceptType == null) ? 0 : acceptType.hashCode());
		result = prime * result + this.httpMethod;
		result = prime * result + ((this.path == null) ? 0 : this.path.hashCode());
		return result;
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		Route other = (Route) obj;
		String acceptType = this.acceptType;
		if (acceptType == null) {
			if (other.acceptType != null) {
				return false;
			}
		} else if (!acceptType.equals(other.acceptType)) {
			return false;
		}
		if (this.httpMethod != other.httpMethod) {
			return false;
		}
		if (this.path == null) {
			if (other.path != null) {
				return false;
			}
		} else if (!this.path.equals(other.path)) {
			return false;
		}
		return true;
	}

}
