/*
 * Java
 *
 * Copyright 2009-2022 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.hoka.http;

import static ej.hoka.http.HttpConstants.EMPTY;

import java.io.IOException;
import java.net.InetAddress;
import java.util.List;

import javax.net.ServerSocketFactory;

import ej.annotation.Nullable;
import ej.hoka.http.encoding.ContentEncoding;
import ej.hoka.http.encoding.EncodingRegistry;
import ej.hoka.http.requesthandler.RequestHandler;
import ej.hoka.http.requesthandler.StaticFilesHandler;
import ej.hoka.http.support.Mime;
import ej.hoka.log.HokaLogger;
import ej.hoka.tcp.TcpServer;

/**
 * HTTP Server.
 * <p>
 * <b>Features + limitations: </b>
 * <ul>
 * <li>CLDC 1.1</li>
 * <li>No fixed configuration files, logging, authorization, encryption.</li>
 * <li>Supports parameter parsing of GET and POST methods</li>
 * <li>Supports both dynamic content and file serving</li>
 * <li>Never caches anything</li>
 * <li>Doesn't limit bandwidth, request time or simultaneous connections</li>
 * <li>Contains a built-in list of most common MIME types</li>
 * <li>All header names are converted to lower case</li>
 * <li>Keep-alive is not supported</li>
 * </ul>
 * <p>
 * <b>Example:</b>
 *
 * <pre>
 * // get a new server which uses a ResourceRequestHandler
 * HttpServer http = HttpServer.builder() //
 * 		.port(8080) //
 * 		.simultaneousConnections(3) //
 * 		.workerCount(3) //
 * 		.build();
 *
 * // register a route
 * http.get("/", requestHandler);
 *
 * // start the server
 * http.start();
 * </pre>
 * <p>
 * To parse the request and write the response, a buffer size of 4096 by default is used. To change this value, set the
 * property "hoka.buffer.size".
 *
 */
public class HttpServer {

	/**
	 * The default timeout duration, here set to 60s.
	 */
	private static final int DEFAULT_TIMEOUT = 60000;

	/**
	 * The underlying TCP server.
	 */
	private final TcpServer server;

	/**
	 * Number of worker to process HTTP requests
	 */
	private final int workerCount;

	/**
	 * Request router to dispatch incoming request to their mapped handler
	 */
	private final RouteHandler routesHandler;

	/**
	 * Registry for supported encodings
	 */
	private final EncodingRegistry encodingRegistry;

	/**
	 * not found exception message and mime type
	 */
	private String notFoundErrorMessage = EMPTY;
	private String notFoundErrorContentType = Mime.MIME_PLAINTEXT;

	/**
	 * internal server exception message and mime type
	 */
	private String internalServerErrorMessage = EMPTY;
	private String internalServerErrorContentType = Mime.MIME_PLAINTEXT;

	/**
	 * Array of {@link Thread} for the session jobs.
	 */
	@Nullable
	private Thread[] workers;

	/**
	 * Activate development mode to send Java exception stack trace to the client.
	 */
	private final boolean devMode;

	/**
	 * Activate strict encoding support. RFC2616 14.3
	 */
	private final boolean strictAcceptEncoding;

	/**
	 * Constructs a HTTP server that manage workers to process the connections from <code>tcpServer</code> with
	 * <code>requestHandler</code>.
	 *
	 * @param port
	 * @param maxSimultaneousConnection
	 * @param workersCount
	 * @param serverSocketFactory
	 * @param connectionTimeout
	 * @param encodingRegistry
	 * @param apiBase
	 * @param staticFilesHandler
	 * @param strictAcceptEncoding
	 * @param trailingSlashSupport
	 * @param devMode
	 */
	private HttpServer(int port, int maxSimultaneousConnection, final int workersCount, // NOSONAR
			ServerSocketFactory serverSocketFactory, int connectionTimeout, final EncodingRegistry encodingRegistry,
			String apiBase, @Nullable StaticFilesHandler staticFilesHandler, boolean strictAcceptEncoding,
			boolean trailingSlashSupport, boolean devMode) {

		this.server = new TcpServer(port, maxSimultaneousConnection, serverSocketFactory, connectionTimeout);
		this.workerCount = workersCount;
		this.encodingRegistry = encodingRegistry;
		this.routesHandler = new RouteHandler(apiBase, staticFilesHandler, trailingSlashSupport);
		this.strictAcceptEncoding = strictAcceptEncoding;
		this.devMode = devMode;
	}

	/**
	 * Configure and build an {@link HttpServer} instance.
	 *
	 * @return {@link HttpServerBuilder} instance
	 */
	public static HttpServerBuilder builder() {
		return new HttpServerBuilder();
	}

	/**
	 * {@link HttpServerBuilder} builder.
	 */
	public static class HttpServerBuilder {

		private int port = 0;
		private int simultaneousConnections = 1;
		private int workerCount = 1;
		private String apiBase = EMPTY;
		private ServerSocketFactory ssf = ServerSocketFactory.getDefault();
		private int connectionTimeout = DEFAULT_TIMEOUT;
		private EncodingRegistry encodingRegistry = new EncodingRegistry();
		@Nullable
		private StaticFilesHandler staticFilesHandler = null;
		private boolean devMode = false;
		private boolean strictAcceptEncoding = false;
		private boolean trailingSlashSupport = false;

		private HttpServerBuilder() {
			// no-op
		}

		/**
		 * Set the server port number.
		 *
		 * @param port
		 *            server port. use 0 set setup a random available port.
		 *
		 * @return {@link HttpServerBuilder} to continue setting up the server parameters
		 * @throws IllegalArgumentException
		 *             if port is {@code < 0}
		 */
		public HttpServerBuilder port(int port) {
			if (port < 0) {
				throw new IllegalArgumentException();
			}
			this.port = port;
			return this;
		}

		/**
		 * Set the max simultaneous connections.
		 *
		 * @param simultaneousConnections
		 *            max simultaneous connections that can be handled by the server.
		 * @return {@link HttpServerBuilder} to continue setting up the server parameters
		 * @throws IllegalArgumentException
		 *             - if simultaneousConnections is {@code <= 0};
		 *
		 */
		public HttpServerBuilder simultaneousConnections(int simultaneousConnections) {
			if (this.simultaneousConnections <= 0) {
				throw new IllegalArgumentException();
			}
			this.simultaneousConnections = simultaneousConnections;
			return this;
		}

		/**
		 * Set threads count that handles incoming request.
		 *
		 * @param workerCount
		 *            number of threads to handle incoming requests
		 * @return {@link HttpServerBuilder} to continue setting up the server parameters
		 * @throws IllegalArgumentException
		 *             - if workerCount is {@code <= 0};
		 */
		public HttpServerBuilder workerCount(int workerCount) {
			if (workerCount <= 0) {
				throw new IllegalArgumentException();
			}
			this.workerCount = workerCount;
			return this;
		}

		/**
		 *
		 * Add a path common base that will be appended to all routes mapped without starting with a forward slash '/'.
		 *
		 * This is useful when dealing with rest API to set the rest api URI base once.
		 *
		 * The base should start with a '/' and also ends with a '/'.
		 *
		 * @param apiBase
		 *            the api base path
		 * @return {@link HttpServerBuilder} to continue setting up the server parameters
		 * @throws IllegalArgumentException
		 *             if apiBase is null or empty
		 */
		public HttpServerBuilder apiBase(@Nullable String apiBase) {
			if (apiBase == null || apiBase.isEmpty()) {
				throw new IllegalArgumentException();
			}

			this.apiBase = apiBase;
			return this;
		}

		/**
		 * Set Server Socket Factory. this used to set ssl
		 *
		 * @param ssf
		 *            Server socket factory to setup HTTPS
		 * @return {@link HttpServerBuilder} to continue setting up the server parameters
		 * @throws IllegalArgumentException
		 *             if ServerSocketFactory parameter is null
		 *
		 */
		public HttpServerBuilder secure(@Nullable ServerSocketFactory ssf) {
			if (ssf == null) {
				throw new IllegalArgumentException();
			}
			this.ssf = ssf;
			return this;
		}

		/**
		 * By default, server is configured to keep connection open during one minute if possible.
		 *
		 * A timeout of zero is interpreted as an infinite timeout.
		 *
		 * @param connectionTimeout
		 *            the timeout duration.
		 * @return {@link HttpServerBuilder} to continue setting up the server parameters
		 * @throws IllegalArgumentException
		 *             if connectionTimeout {@code < 0 }
		 */
		public HttpServerBuilder connectionTimeout(int connectionTimeout) {
			if (connectionTimeout < 0) {
				throw new IllegalArgumentException();
			}
			this.connectionTimeout = connectionTimeout;
			return this;
		}

		/**
		 * If this method is not called an instance of {@link EncodingRegistry} is used by the server.
		 *
		 * @param encodingRegistry
		 *            the registry of available encoding handlers.
		 * @return {@link HttpServerBuilder} to continue setting up the server parameters
		 * @throws IllegalArgumentException
		 *             if encodingRegistry is null
		 */
		public HttpServerBuilder encodingRegistry(@Nullable EncodingRegistry encodingRegistry) {
			if (encodingRegistry == null) {
				throw new IllegalArgumentException();
			}
			this.encodingRegistry = encodingRegistry;
			return this;
		}

		/**
		 * RFC2616 14.3
		 *
		 * Activate strict content encoding acceptance.
		 *
		 * The server will returns a 406 (Not Acceptable) for every request with a header Accept-Encoding with no
		 * encoding supported by the server.
		 *
		 * Register custom content encoding using {@link EncodingRegistry#registerContentEncoding(ContentEncoding)}
		 *
		 * @return {@link HttpServerBuilder} to continue setting up the server parameters
		 */
		public HttpServerBuilder withStrictAcceptContentEncoding() {
			this.strictAcceptEncoding = true;
			return this;
		}

		/**
		 *
		 * Activate trailing slash matching support in URI.
		 *
		 * When used URI '/hello' and '/hello/' will point to different request handler
		 *
		 * otherwise URI '/hello' and '/hello/' are the same
		 *
		 * @return {@link HttpServerBuilder} to continue setting up the server parameters
		 */
		public HttpServerBuilder withTrailingSlashSupport() {
			this.trailingSlashSupport = true;
			return this;
		}

		/**
		 * Set the static files handler to serve static files.
		 *
		 * Use this only for public resources.
		 *
		 * @param handler
		 *            the {@link StaticFilesHandler} instance
		 * @return {@link HttpServerBuilder} to continue setting up the server parameters
		 * @throws IllegalArgumentException
		 *             if the handler is null
		 */
		public HttpServerBuilder staticFilesHandler(@Nullable StaticFilesHandler handler) {
			if (handler == null) {
				throw new IllegalArgumentException();
			}
			this.staticFilesHandler = handler;
			return this;
		}

		/**
		 * Activate development mode. The server will send stack trace of thrown exceptions to the clients.
		 *
		 * @return {@link HttpServerBuilder} to continue setting up the server parameters
		 */
		public HttpServerBuilder developmentMode() {
			this.devMode = true;
			return this;
		}

		/**
		 * Builds and returns an instance of {@link HttpServer} with the configured parameters.
		 *
		 * @return an instance of {@link HttpServer} with the configured parameters.
		 */
		public HttpServer build() {
			return new HttpServer(this.port, this.simultaneousConnections, this.workerCount, this.ssf,
					this.connectionTimeout, this.encodingRegistry, this.apiBase, this.staticFilesHandler,
					this.strictAcceptEncoding, this.trailingSlashSupport, this.devMode);
		}
	}

	/**
	 * Start the {@link HttpServer} (in a dedicated thread): start listening for connections and start workers to
	 * process opened connections.
	 * <p>
	 * Multiple start is not allowed.
	 *
	 * @throws IOException
	 *             if an error occurs during the creation of the socket.
	 */
	public void start() throws IOException {

		this.server.start();

		Thread[] ws;
		this.workers = ws = new Thread[this.workerCount];
		for (int i = this.workerCount - 1; i >= 0; i--) {
			final Thread worker = new Thread(new Worker(this.server, this.routesHandler, this.encodingRegistry,
					this.strictAcceptEncoding, this.notFoundErrorMessage, this.notFoundErrorContentType,
					this.internalServerErrorMessage, this.internalServerErrorContentType, this.devMode),
					"HTTP-Worker-" + i); //$NON-NLS-1$
			ws[i] = worker;
			worker.start();
		}
	}

	/**
	 * Stops the {@link HttpServer}. Stops listening for connections. This method blocks until all session jobs are
	 * stopped.
	 */
	public void stop() {

		this.server.stop();

		final Thread[] jobs = this.workers;
		if (jobs != null) {
			for (int i = jobs.length - 1; i >= 0; i--) {
				try {
					jobs[i].join();
				} catch (InterruptedException e) {
					HokaLogger.instance.error(e);
					jobs[i].interrupt(); // Restore interrupted state
				}
			}
		}
	}

	/**
	 * Map the handler to HTTP GET requests on the path.
	 *
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public void get(final String path, final RequestHandler handler) {
		get(path, null, handler);
	}

	/**
	 * Map the handler to HTTP GET requests on the path.
	 *
	 * @param path
	 *            request path
	 * @param acceptType
	 *            accepted type
	 * @param handler
	 *            request handler
	 */
	public void get(final String path, @Nullable final String acceptType, final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.GET, path, acceptType, handler);
	}

	/**
	 * Map the handler to HTTP POST requests on the path.
	 *
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public void post(final String path, final RequestHandler handler) {
		post(path, null, handler);
	}

	/**
	 * Map the handler to HTTP POST requests on the path.
	 *
	 * @param path
	 *            request path
	 * @param acceptType
	 *            accepted type
	 * @param handler
	 *            request handler
	 */
	public void post(final String path, @Nullable final String acceptType, final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.POST, path, acceptType, handler);
	}

	/**
	 * Map the handler to HTTP PUT requests on the path.
	 *
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public void put(final String path, final RequestHandler handler) {
		put(path, null, handler);
	}

	/**
	 * Map the handler to HTTP PUT requests on the path.
	 *
	 * @param path
	 *            request path
	 * @param acceptType
	 *            accepted type
	 * @param handler
	 *            request handler
	 */
	public void put(final String path, @Nullable final String acceptType, final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.PUT, path, acceptType, handler);
	}

	/**
	 * Map the handler to HTTP POST requests on the path.
	 *
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public void delete(final String path, final RequestHandler handler) {
		delete(path, null, handler);
	}

	/**
	 * Map the handler to HTTP DELETE requests on the path.
	 *
	 * @param path
	 *            request path
	 * @param acceptType
	 *            accepted type
	 * @param handler
	 *            request handler
	 */
	public void delete(final String path, @Nullable final String acceptType, final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.DELETE, path, acceptType, handler);
	}

	/**
	 * Map the handler to HTTP HEAD requests on the path.
	 *
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public void head(final String path, final RequestHandler handler) {
		head(path, null, handler);
	}

	/**
	 * Map the handler to HTTP HEAD requests on the path.
	 *
	 * @param path
	 *            request path
	 * @param acceptType
	 *            accepted type
	 * @param handler
	 *            request handler
	 */
	public void head(final String path, @Nullable final String acceptType, final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.HEAD, path, acceptType, handler);
	}

	/**
	 * Map the handler to HTTP CONNECT requests on the path.
	 *
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public void connect(final String path, final RequestHandler handler) {
		connect(path, null, handler);
	}

	/**
	 * Map the handler to HTTP CONNECT requests on the path.
	 *
	 * @param path
	 *            request path
	 * @param acceptType
	 *            accepted type
	 * @param handler
	 *            request handler
	 */
	public void connect(final String path, @Nullable final String acceptType, final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.CONNECT, path, acceptType, handler);
	}

	/**
	 * Map the handler to HTTP OPTIONS requests on the path.
	 *
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public void options(final String path, final RequestHandler handler) {
		options(path, null, handler);
	}

	/**
	 * Map the handler to HTTP OPTIONS requests on the path.
	 *
	 * @param path
	 *            request path
	 * @param acceptType
	 *            accepted type
	 * @param handler
	 *            request handler
	 */
	public void options(final String path, @Nullable final String acceptType, final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.OPTIONS, path, acceptType, handler);
	}

	/**
	 * Map the handler to HTTP TRACE requests on the path.
	 *
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public void trace(final String path, final RequestHandler handler) {
		trace(path, null, handler);
	}

	/**
	 * Map the handler to HTTP TRACE requests on the path.
	 *
	 * @param path
	 *            request path
	 * @param acceptType
	 *            accepted type
	 * @param handler
	 *            request handler
	 */
	public void trace(final String path, @Nullable final String acceptType, final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.TRACE, path, acceptType, handler);
	}

	/**
	 * Map the handler to HTTP PATCH requests on the path.
	 *
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public void patch(final String path, final RequestHandler handler) {
		patch(path, null, handler);
	}

	/**
	 * Map the handler to HTTP PATCH requests on the path.
	 *
	 * @param path
	 *            request path
	 * @param acceptType
	 *            accepted type
	 * @param handler
	 *            request handler
	 */
	public void patch(final String path, @Nullable final String acceptType, final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.PATCH, path, acceptType, handler);
	}

	/**
	 * Map the handler as a before filter on the path.
	 *
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public void before(String path, final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.BEFORE, path, handler);
	}

	/**
	 * Map the handler as a before filter on all paths.
	 *
	 * @param handler
	 *            request handler
	 */
	public void before(final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.BEFORE_ALL, RouteHandler.ALL_PATH, handler);
	}

	/**
	 * Map the handler as an after filter on the path.
	 *
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public void after(String path, final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.AFTER, path, handler);
	}

	/**
	 * Map the handler as an after filter on all paths.
	 *
	 * @param handler
	 *            request handler
	 */
	public void after(final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.AFTER_ALL, RouteHandler.ALL_PATH, handler);
	}

	/**
	 * Immediately Stop the execution and return a 200 OK HTTP response.
	 */
	public static final void halt() {
		throw new HaltException();
	}

	/**
	 * immediately Stop the execution and return an HTTP response with the status.
	 *
	 * @param status
	 *            http status
	 */
	public static final void halt(String status) {
		throw new HaltException(status);
	}

	/**
	 * immediately Stop the execution and return an HTTP response with the status and body.
	 *
	 * @param status
	 *            http status
	 * @param body
	 *            response body
	 */
	public static final void halt(String status, String body) {
		throw new HaltException(status, body);
	}

	/**
	 * Override not found error 404.
	 *
	 * @param response
	 *            error message returned as text/plain
	 */
	public final void notFoundError(@Nullable final String response) {
		if (response == null) {
			throw new IllegalArgumentException();
		}
		notFoundError(response, Mime.MIME_PLAINTEXT);
	}

	/**
	 * Override not found error 404.
	 *
	 * @param response
	 *            error message
	 * @param contentType
	 *            response content type. application/json for example
	 */
	public final void notFoundError(@Nullable final String response, @Nullable final String contentType) {
		if (response == null || contentType == null) {
			throw new IllegalArgumentException();
		}
		this.notFoundErrorMessage = response;
		this.notFoundErrorContentType = contentType;
	}

	/**
	 * Override internal server error message 501.
	 *
	 * @param response
	 *            error message returned as text/plain
	 */
	public final void internalServerError(@Nullable final String response) {
		if (response == null) {
			throw new IllegalArgumentException();
		}
		internalServerError(response, Mime.MIME_PLAINTEXT);
	}

	/**
	 * Override internal server error message 501.
	 *
	 * @param response
	 *            error message
	 * @param contentType
	 *            response content type. application/json for example
	 */
	public final void internalServerError(@Nullable final String response, @Nullable final String contentType) {
		if (response == null || contentType == null) {
			throw new IllegalArgumentException();
		}
		this.internalServerErrorMessage = response;
		this.internalServerErrorContentType = contentType;
	}

	/**
	 * Map a handler to a specific exception.
	 *
	 * @param exception
	 *            exception to handle
	 * @param handler
	 *            exception handler
	 */
	public final void exception(@Nullable Class<? extends Exception> exception, @Nullable RequestHandler handler) {
		if (exception == null || handler == null) {
			throw new IllegalArgumentException();
		}
		this.routesHandler.addExceptionHandler(exception, handler);
	}

	/**
	 * Gets the list of registered routes.
	 *
	 * @return list of registered routes
	 */
	public final List<Route> getRoutes() {
		return this.routesHandler.getRoutes();
	}

	/**
	 * Gets the port number to which this server is listening to, if any.
	 *
	 * @return the port number to which this server is listening or-1 if the socket is not bound yet.
	 */
	public int getPort() {
		return this.server.getPort();
	}

	/**
	 * Gets the address to which this server is bound, if any.
	 *
	 * @return the address to which this server is bound,or null if the socket is unbound.
	 */
	@Nullable
	public InetAddress getInetAddress() {
		return this.server.getInetAddress();
	}

}
