/*
 * Java
 *
 * Copyright 2020-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.library.iot.rcommand.bluetooth;

import java.util.HashMap;
import java.util.Map;

import ej.annotation.Nullable;
import ej.bluetooth.BluetoothAttribute;
import ej.bluetooth.BluetoothCharacteristic;
import ej.bluetooth.BluetoothDescriptor;
import ej.bluetooth.BluetoothService;

public class BluetoothDatabase {

	private final Map<BluetoothService, Short> services;
	private final Map<BluetoothAttribute, Short> attributes;

	public BluetoothDatabase() {
		this.services = new HashMap<>();
		this.attributes = new HashMap<>();
	}

	public void clear() {
		this.services.clear();
		this.attributes.clear();
	}

	public @Nullable Short getServiceHandle(BluetoothService service) {
		return this.services.get(service);
	}

	public @Nullable Short getAttributeHandle(BluetoothAttribute attribute) {
		return this.attributes.get(attribute);
	}

	public @Nullable BluetoothAttribute getAttribute(short attributeHandle) {
		for (Map.Entry<BluetoothAttribute, Short> entry : this.attributes.entrySet()) {
			Short handle = entry.getValue();
			assert (handle != null);
			if (handle.shortValue() == attributeHandle) {
				return entry.getKey();
			}
		}
		return null;
	}

	public short addService(BluetoothService service, short counter, @Nullable short[] handles) {
		int handleIndex = 0;

		short serviceHandle = counter;
		counter++;
		if (handles != null) {
			handles[handleIndex] = serviceHandle;
			handleIndex++;
		}
		this.services.put(service, Short.valueOf(serviceHandle));

		int numCharacteristics = service.getNumCharacteristics();
		for (int c = 0; c < numCharacteristics; c++) {
			BluetoothCharacteristic characteristic = service.getCharacteristic(c);
			short characteristicHandle = counter;
			counter++;
			if (handles != null) {
				handles[handleIndex] = characteristicHandle;
				handleIndex++;
			}
			this.attributes.put(characteristic, Short.valueOf(characteristicHandle));

			int numDescriptors = characteristic.getNumDescriptors();
			for (int d = 0; d < numDescriptors; d++) {
				BluetoothDescriptor descriptor = characteristic.getDescriptor(d);
				short descriptorHandle = counter;
				counter++;
				if (handles != null) {
					handles[handleIndex] = descriptorHandle;
					handleIndex++;
				}
				this.attributes.put(descriptor, Short.valueOf(descriptorHandle));
			}
		}

		return counter;
	}
}
