/*
 * Java
 *
 * Copyright 2010-2025 MicroEJ Corp.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.microui.display;

import ej.microui.MicroUIException;

/**
 * An <code>Image</code> represents a graphical two-dimensional object. An image is divided in pixels, and each pixel
 * holds a value defining its color.
 * <p>
 * This class provides a static method in order to retrieve an image from a resource which is in the internal image
 * format defined by the MicroUI implementation.
 */
public class Image {

	/**
	 * Forbidden constructor: call {@link #getImage(String)} to get an instance of {@link Image}.
	 */
	/* default */ Image() {
	}

	/**
	 * Returns whether an image can be retrieved from a resource.
	 * <p>
	 * This method may be used to know whether calling {@link Image#getImage(String)} with the given path would be
	 * successful or not.
	 *
	 * @param path
	 *            the resource path.
	 * @return <code>true</code> if the image can be retrieved, <code>false</code> otherwise.
	 * @throws MicroUIException
	 *             if the resource path doesn't start with "/".
	 * @throws MicroUIException
	 *             if MicroUI is not started.
	 * @throws SecurityException
	 *             if a security manager exists and does not allow the caller to load an image.
	 * @see #getImage(String)
	 * @since 3.0
	 */
	public static boolean canGetImage(String path) {
		throw new RuntimeException();
	}

	/**
	 * Gets an immutable image from a resource.
	 * <p>
	 * This method can only retrieve images which are in the internal image format defined by the MicroUI
	 * implementation. This operation does not require a dynamic allocation in order to store the image pixels.
	 * <p>
	 * If the resource cannot be retrieved, a {@link MicroUIException} is thrown even if the resource is present but
	 * requires a loading step.
	 *
	 * @param path
	 *            the resource path.
	 * @return the retrieved image.
	 * @throws MicroUIException
	 *             if the image could not be retrieved for any reason (see {@link MicroUIException#getErrorCode()}).
	 * @throws MicroUIException
	 *             if the resource path doesn't start with "/".
	 * @throws MicroUIException
	 *             if MicroUI is not started.
	 * @throws SecurityException
	 *             if a security manager exists and does not allow the caller to retrieve an image.
	 * @since 3.0
	 */
	public static Image getImage(String path) {
		throw new RuntimeException();
	}

	/**
	 * Returns the width of the image.
	 *
	 * @return the width of the image, in pixels.
	 */
	public int getWidth() {
		throw new RuntimeException();
	}

	/**
	 * Returns the height of the image.
	 *
	 * @return the height of the image, in pixels.
	 *
	 */
	public int getHeight() {
		throw new RuntimeException();
	}

	/**
	 * Returns whether this image is transparent.
	 * <p>
	 * An image is considered as transparent if it contains at least one transparent pixel.
	 *
	 * @return <code>true</code> if the image is transparent, <code>false</code> otherwise.
	 */
	public boolean isTransparent() {
		throw new RuntimeException();
	}

	/**
	 * Returns the color of a pixel of this image.
	 * <p>
	 * For more information on the format of the color value returned by this method, refer to the first paragraphs of
	 * the {@link GraphicsContext#readPixels centralized pixel reading documentation}.
	 *
	 * @param x
	 *            the x coordinate of the pixel.
	 * @param y
	 *            the y coordinate of the pixel.
	 * @return the color of the pixel, in ARGB format.
	 * @throws IllegalArgumentException
	 *             if the given pixel coordinates are out of the bounds of this image.
	 * @throws MicroUIException
	 *             if this image has been closed (see {@link ResourceImage#close()}).
	 */
	public int readPixel(int x, int y) {
		throw new RuntimeException();
	}

	/**
	 * Retrieves the color of the pixels of a region of this image.
	 * <p>
	 * For more information on this method, refer to the {@link GraphicsContext#readPixels centralized pixel reading
	 * documentation}.
	 *
	 * @param array
	 *            the array in which the pixel data should be stored.
	 * @param offset
	 *            the index of the array at which the first pixel color should be stored.
	 * @param scanLength
	 *            the relative offset in the array between two corresponding pixels in consecutive rows.
	 * @param x
	 *            the x-coordinate of the top-left pixel of the region.
	 * @param y
	 *            the y-coordinate of the top-left pixel of the region.
	 * @param width
	 *            the width of the region.
	 * @param height
	 *            the height of the region.
	 * @throws ArrayIndexOutOfBoundsException
	 *             if the array is not big enough to hold the color of all the pixels of the region.
	 * @throws IllegalArgumentException
	 *             if a part of the region is out of the bounds of this image.
	 * @throws IllegalArgumentException
	 *             if the absolute value of <code>scanLength</code> is lower than <code>width</code>.
	 */
	public void readPixels(int[] array, int offset, int scanLength, int x, int y, int width, int height) {
		throw new RuntimeException();
	}

	/**
	 * Returns the SNI context data of this image.
	 * <p>
	 * The SNI context can be used to call a native method with SNI. This allows to identify and to use an image in the
	 * native world in order to perform drawings on the image (if it is mutable) or to use it as a source.
	 * <p>
	 * If the image is mutable, the data returned by this method is the same than the data returned when getting the SNI
	 * context of the graphics context of the image (see {@link GraphicsContext#getSNIContext()}).
	 * <p>
	 * If the SNI context is used in a native method, the application should synchronize the calls on the image object
	 * in order to prevent the image from being closed during the drawing.
	 * <p>
	 * The data format is implementation specific.
	 *
	 * @return the SNI context of this image.
	 * @throws MicroUIException
	 *             if this image has been closed (see {@link ResourceImage#close()}).
	 */
	public byte[] getSNIContext() {
		throw new RuntimeException();
	}
}
