/*
 * Java
 *
 * Copyright 2022-2026 MicroEJ Corp.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import static org.junit.Assert.assertEquals;

import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;
import ej.microvg.BufferedVectorImage;
import ej.microvg.Matrix;
import ej.microvg.Path;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorImage;

/**
 * Tests the drawing of MicroVG shapes into a BufferedVectorImage and using clip(s).
 */
@SuppressWarnings("nls")
public class TestBufferedVectorImageClip {

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		g.reset();
		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());
	}

	/**
	 * Tests no clip
	 */
	@Test
	public static void testNoClip() {

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		BufferedVectorImage image = new BufferedVectorImage(20, 20);
		GraphicsContext ig = image.getGraphicsContext();

		g.translate(200, 200);

		ig.setColor(Colors.RED);
		drawPath(ig);

		VectorGraphicsPainter.drawImage(g, image, new Matrix());
		display.flush();

		// source: 20 x 20
		// dest: 20 x 20 at (200,200)
		checkAreas(0, 0, 20, 20);
	}

	/**
	 * Tests clip on destination
	 */
	@Test
	public static void testClipOnDestination() {

		int x = 1;
		int y = 1;
		int w = 10;
		int h = 10;

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(2 * w, 2 * h)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);
			g.setClip(x, y, w, h);

			ig.setColor(Colors.RED);
			drawPath(ig);

			VectorGraphicsPainter.drawImage(g, image, new Matrix());
			display.flush();

			// source: 20 x 20
			// dest: 10 x 10 at (201,201)
			checkAreas(x, y, w, h);
		}
	}

	/**
	 * Tests outside clip on destination
	 */
	@Test
	public static void testOutsideClipOnDestination() {

		int w = 10;
		int h = 10;

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(w, h)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);
			g.setClip(w + 1, 0, w, h);

			ig.setColor(Colors.RED);
			drawPath(ig);

			VectorGraphicsPainter.drawImage(g, image, new Matrix());
			display.flush();

			checkAreas(0, 0, w, h, TestUtilities.BACKGROUND_COLOR);
		}
	}

	/**
	 * Tests clip on source
	 */
	@Test
	public static void testClipOnSource() {

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(20, 20)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);

			ig.setColor(Colors.RED);
			ig.setClip(2, 2, 5, 5);
			drawPath(ig);

			VectorGraphicsPainter.drawImage(g, image, new Matrix());
			display.flush();

			// source: 5 x 5
			// dest: 5 x 5 at (202,202)
			checkAreas(2, 2, 5, 5);
		}
	}

	/**
	 * Tests outside clip on source
	 */
	@Test
	public static void testOutsideClipOnSource() {

		int w = 10;
		int h = 10;

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(w, h)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);

			ig.setColor(Colors.RED);
			ig.setClip(w + 1, 0, w, h);
			drawPath(ig);

			VectorGraphicsPainter.drawImage(g, image, new Matrix());
			display.flush();

			checkAreas(0, 0, w, h, TestUtilities.BACKGROUND_COLOR);
		}
	}

	/**
	 * Tests clip on source and destination
	 */
	@Test
	public static void testClipOnSourceAndDestination() {

		int x = 1;
		int y = 10;
		int w = 25;
		int h = 5;

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(60, 60)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);
			g.setClip(x, y, w, h);

			ig.setColor(Colors.RED);
			ig.setClip(5, 5, 10, 10);
			drawPath(ig);

			VectorGraphicsPainter.drawImage(g, image, new Matrix());
			display.flush();

			// source: 10 x 10
			// dest: 10 x 5 at (205,210)
			checkAreas(5, y, 10, 5);
		}
	}

	/**
	 * Tests scaling on source
	 */
	@Test
	public static void testScalingOnSource() {

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(20, 20)) {
			GraphicsContext ig = image.getGraphicsContext();
			g.translate(200, 200);

			ig.setColor(Colors.RED);
			drawPath(ig);

			Matrix matrix = new Matrix();
			matrix.setScale(0.25f, 0.25f);
			VectorGraphicsPainter.drawImage(g, image, matrix);
			display.flush();

			// source: 20 x 20
			// dest: 5 x 5 at (200,200)
			checkAreas(0, 0, 5, 5);
		}
	}

	/**
	 * Tests scaling on source and clip on destination
	 */
	@Test
	public static void testScalingOnSourceAndClipOnDestination() {

		int x = 1;
		int y = 1;
		int w = 10;
		int h = 10;

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(20, 20)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);
			g.setClip(x, y, w, h);

			ig.setColor(Colors.RED);
			drawPath(ig);

			Matrix matrix = new Matrix();
			matrix.setScale(0.25f, 0.25f);
			VectorGraphicsPainter.drawImage(g, image, matrix);
			display.flush();

			// source: 20 x 20
			// dest: 4 x 4 at (201,201)
			checkAreas(x, y, 4, 4);
		}
	}

	/**
	 * Tests scaling and clip on source
	 */
	@Test
	public static void testScalingAndClipOnSource() {

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(60, 60)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);

			ig.setColor(Colors.RED);
			ig.setClip(12, 12, 32, 32);
			drawPath(ig);

			Matrix matrix = new Matrix();
			matrix.setScale(0.25f, 0.25f);
			VectorGraphicsPainter.drawImage(g, image, matrix);
			display.flush();

			// source: 60 x 60
			// dest: 8 x 8 at (203,203) // 3 = 12 / 4
			checkAreas(3, 3, 8, 8);
		}
	}

	/**
	 * Tests scaling on source and clip on source and destination
	 */
	@Test
	public static void testScalingOnSourceAndClipOnSourceAndDestination() {

		int x = 1;
		int y = 1;
		int w = 5;
		int h = 5;

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(60, 60)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);
			g.setClip(x, y, w, h);

			ig.setColor(Colors.RED);
			ig.setClip(12, 12, 32, 32);
			drawPath(ig);

			Matrix matrix = new Matrix();
			matrix.setScale(0.25f, 0.25f);
			VectorGraphicsPainter.drawImage(g, image, matrix);
			display.flush();

			// source: 60 x 60
			// dest: 3 x 3 at (203,203) // 3 = 12 / 4
			checkAreas(3, 3, 3, 3);
		}
	}

	/**
	 * Tests rotation on source
	 */
	@Test
	public static void testRotationOnSource() {

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(16, 8)) {
			GraphicsContext ig = image.getGraphicsContext();
			g.setTranslation(200, 200);

			ig.setColor(Colors.RED);
			drawPath(ig);

			Matrix matrix = new Matrix();
			matrix.setRotate(90);
			VectorGraphicsPainter.drawImage(g, image, matrix, 0xff);
			display.flush();

			// source: 16 x 8
			// dest: 8 x 16 at (192,200)
			checkAreas((int) -image.getHeight(), 0, (int) image.getHeight(), (int) image.getWidth());
		}
	}

	/**
	 * Tests rotation on source and clip on destination
	 */
	@Test
	public static void testRotationOnSourceAndClipOnDestination() {

		int x = 1;
		int y = 1;
		int w = 5;
		int h = 5;

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(16, 8)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200 - (int) image.getHeight(), 200);
			g.setClip(x, y, w, h);
			g.translate((int) image.getHeight(), 0);

			ig.setColor(Colors.RED);
			drawPath(ig);

			Matrix matrix = new Matrix();
			matrix.setRotate(90);
			VectorGraphicsPainter.drawImage(g, image, matrix);
			display.flush();

			// source: 16 x 8
			// dest: 5 x 5 at (193,201)
			checkAreas(x - (int) image.getHeight(), y, w, h);
		}
	}

	/**
	 * Tests rotation and clip on source
	 */
	@Test
	public static void testRotationAndClipOnSource() {

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(30, 20)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);

			ig.setColor(Colors.RED);
			ig.setClip(2, 2, 16, 8);
			drawPath(ig);

			Matrix matrix = new Matrix();
			matrix.setRotate(90);
			VectorGraphicsPainter.drawImage(g, image, matrix);
			display.flush();

			// source: 30 x 20
			// dest: 8 x 16 at (190,202)
			checkAreas(-8 - 2, 2, 8, 16);
		}
	}

	/**
	 * Tests rotation on source and clip on source and destination
	 */
	@Test
	public static void testRotationOnSourceAndClipOnSourceAndDestination() {

		int x = 1;
		int y = 1;
		int w = 5;
		int h = 5;

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(60, 40)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200 - (int) image.getHeight(), 200);
			g.setClip(x, y, w, h);
			g.translate((int) image.getHeight(), 0);

			ig.setColor(Colors.RED);
			ig.setClip(2, 2, (int) image.getWidth() - 4, (int) image.getHeight() - 4);
			drawPath(ig);

			Matrix matrix = new Matrix();
			matrix.setRotate(90);
			VectorGraphicsPainter.drawImage(g, image, matrix);
			display.flush();

			// source: 60 x 40
			// dest: 4 x 4 at (162,202)
			checkAreas(-(int) image.getHeight() + 2, 2, w - 1, h - 1);
		}
	}

	/**
	 * Tests raw image with clip on destination
	 */
	@Test
	public static void testImageClipOnDestination() {

		int x = 1;
		int y = 1;
		int w = 10;
		int h = 10;

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(2 * w, 2 * h)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);
			g.setClip(x, y, w, h);

			VectorGraphicsPainter.drawImage(ig, getImage(20, 20), new Matrix());

			VectorGraphicsPainter.drawImage(g, image, new Matrix());
			display.flush();

			// source: 20 x 20
			// dest: 10 x 10 at (201,201)
			checkAreas(x, y, w, h);
		}
	}

	/**
	 * Tests raw image with clip on source
	 */
	@Test
	public static void testImageClipOnSource() {

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(20, 20)) {

			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);

			ig.setColor(Colors.RED);
			ig.setClip(2, 2, 5, 5);
			VectorGraphicsPainter.drawImage(ig, getImage(20, 20), new Matrix());

			VectorGraphicsPainter.drawImage(g, image, new Matrix());
			display.flush();

			// source: 5 x 5
			// dest: 5 x 5 at (202,202)
			checkAreas(2, 2, 5, 5);
		}
	}

	/**
	 * Tests raw image with clip on source and destination
	 */
	@Test
	public static void testImageClipOnSourceAndDestination() {

		int x = 1;
		int y = 10;
		int w = 25;
		int h = 5;

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(60, 60)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);
			g.setClip(x, y, w, h);

			ig.setClip(5, 5, 10, 10);
			VectorGraphicsPainter.drawImage(ig, getImage(60, 60), new Matrix());

			VectorGraphicsPainter.drawImage(g, image, new Matrix());
			display.flush();

			// source: 10 x 10
			// dest: 10 x 5 at (205,210)
			checkAreas(5, y, 10, 5);
		}
	}

	/**
	 * Tests raw image with scaling on source
	 */
	@Test
	public static void testImageScalingOnSource() {

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(20, 20)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);

			VectorGraphicsPainter.drawImage(ig, getImage(20, 20), new Matrix());

			Matrix matrix = new Matrix();
			matrix.setScale(0.25f, 0.25f);
			VectorGraphicsPainter.drawImage(g, image, matrix);
			display.flush();

			// source: 20 x 20
			// dest: 5 x 5 at (200,200)
			checkAreas(0, 0, 5, 5);
		}
	}

	/**
	 * Tests raw image with scaling on source and clip on destination
	 */
	@Test
	public static void testImageScalingOnSourceAndClipOnDestination() {

		int x = 1;
		int y = 1;
		int w = 10;
		int h = 10;

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(2 * w, 2 * h)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);
			g.setClip(x, y, w, h);

			VectorGraphicsPainter.drawImage(ig, getImage(20, 20), new Matrix());

			Matrix matrix = new Matrix();
			matrix.setScale(0.25f, 0.25f);
			VectorGraphicsPainter.drawImage(g, image, matrix);
			display.flush();

			// source: 20 x 20
			// dest: 4 x 4 at (201,201)
			checkAreas(x, y, 4, 4);
		}
	}

	/**
	 * Tests raw image with scaling and clip on source
	 */
	@Test
	public static void testImageScalingAndClipOnSource() {

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(60, 60)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);

			ig.setClip(12, 12, 32, 32);
			VectorGraphicsPainter.drawImage(ig, getImage(60, 60), new Matrix());

			Matrix matrix = new Matrix();
			matrix.setScale(0.25f, 0.25f);
			VectorGraphicsPainter.drawImage(g, image, matrix);
			display.flush();

			// source: 60 x 60
			// dest: 5 x 5 at (203,203) // 3 = 12 / 4
			checkAreas(3, 3, 8, 8);
		}
	}

	/**
	 * Tests raw image with scaling on source and clip on source and destination
	 */
	@Test
	public static void testImageScalingOnSourceAndClipOnSourceAndDestination() {

		int x = 1;
		int y = 1;
		int w = 5;
		int h = 5;

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(60, 60)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);
			g.setClip(x, y, w, h);

			ig.setClip(12, 12, 32, 32);
			VectorGraphicsPainter.drawImage(ig, getImage(60, 60), new Matrix());

			Matrix matrix = new Matrix();
			matrix.setScale(0.25f, 0.25f);
			VectorGraphicsPainter.drawImage(g, image, matrix);
			display.flush();

			// source: 60 x 60
			// dest: 3 x 3 at (203,203) // 3 = 12 / 4
			checkAreas(3, 3, 3, 3);
		}
	}

	/**
	 * Tests several vector images with a non-orthogonal transformation (rotation): to draw a vector image, the clip is
	 * modified. After the drawing, the clip must be restored.
	 */
	@Test
	public static void testSeveralVectorImage() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(50, 50)) {
			GraphicsContext ig = image.getGraphicsContext();

			// fill the BVI
			ig.setColor(Colors.BLUE);
			drawPath(ig);

			// draw VG images: three images are BVI out of bounds
			ig.setTranslation(10, 10);
			VectorGraphicsPainter.drawImage(ig, getImage(20, 20), new Matrix());
			ig.setTranslation(40, 10);
			VectorGraphicsPainter.drawImage(ig, getImage(20, 20), new Matrix());
			ig.setTranslation(10, 40);
			VectorGraphicsPainter.drawImage(ig, getImage(20, 20), new Matrix());
			ig.setTranslation(40, 40);
			VectorGraphicsPainter.drawImage(ig, getImage(20, 20), new Matrix());

			Matrix matrix = new Matrix();
			matrix.setRotate(45);

			// reference: draw BVI without clip (simulate clip with a rectangle)
			g.setTranslation((int) (g.getWidth() / 2 - image.getWidth()),
					(int) (g.getHeight() / 2 - image.getHeight()));
			VectorGraphicsPainter.drawImage(g, image, matrix);
			g.reset();
			Painter.fillRectangle(g, 0, g.getHeight() / 2, g.getWidth() / 2, g.getHeight() / 2);

			// [right] test: draw BVI with clip (lower part in Y must be hidden)
			g.reset();
			g.setClip(g.getWidth() / 2, 0, g.getWidth() / 2, g.getHeight() / 2);
			g.setTranslation((int) (g.getWidth() / 2 + image.getWidth()),
					(int) (g.getHeight() / 2 - image.getHeight()));
			VectorGraphicsPainter.drawImage(g, image, matrix);

			display.flush();

			// source: 60 x 60
			// dest: 72 x 72
			g.reset();
			TestUtilities.compareAreas("clip", g.getWidth() / 2 - 85, g.getHeight() / 2 - 50, g.getWidth() / 2 + 15,
					g.getHeight() / 2 - 50, 72, 72, 0.02f);
		}
	}

	/**
	 * Tests several vector images with a non-orthogonal transformation (rotation): to draw the BVI, the bounds of each
	 * vector image must not be used as clip: have to consider the BVI bounds in globality.
	 */
	@Test
	public static void testVectorImageClip() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(50, 50)) {
			GraphicsContext ig = image.getGraphicsContext();

			// fill the BVI
			ig.setColor(Colors.BLUE);
			drawPath(ig);

			// draw VG images: three images are BVI out of bounds
			ig.setTranslation(10, 10);
			VectorGraphicsPainter.drawImage(ig, getImage(20, 20), new Matrix());
			ig.setTranslation(40, 10);
			VectorGraphicsPainter.drawImage(ig, getImage(20, 20), new Matrix());
			ig.setTranslation(10, 40);
			VectorGraphicsPainter.drawImage(ig, getImage(20, 20), new Matrix());
			ig.setTranslation(40, 40);
			VectorGraphicsPainter.drawImage(ig, getImage(20, 20), new Matrix());

			Matrix matrix = new Matrix();
			matrix.setRotate(45);

			g.reset();
			g.setClip(g.getWidth() / 2, 0, g.getWidth() / 2, g.getHeight() / 2);
			g.setTranslation((int) (g.getWidth() / 2 + image.getWidth()),
					(int) (g.getHeight() / 2 - image.getHeight()));
			VectorGraphicsPainter.drawImage(g, image, matrix);

			display.flush();

			g.reset();
			int gw = g.getWidth() / 2;
			int gh = g.getHeight() / 2;

			TestUtilities.check("image TL pixel TL", gw + 49, gh - 34, g, Colors.RED);
			TestUtilities.check("image TL pixel TR", gw + 61, gh - 22, g, Colors.RED);
			TestUtilities.check("image TL pixel BR", gw + 49, gh - 11, g, Colors.RED);
			TestUtilities.check("image TL pixel BL", gw + 39, gh - 22, g, Colors.RED);

			TestUtilities.check("image TR pixel TL", gw + 71, gh - 13, g, Colors.RED);
			TestUtilities.check("image TR pixel TR", gw + 83, gh - 1, g, Colors.RED);
			TestUtilities.check("image TR pixel BR", gw + 71, gh + 12, g, Colors.BLACK);
			TestUtilities.check("image TR pixel BL", gw + 61, gh - 1, g, Colors.RED);

			TestUtilities.check("image BR pixel TL", gw + 49, gh + 1, g, Colors.BLACK);
			TestUtilities.check("image BR pixel TR", gw + 61, gh + 13, g, Colors.BLACK);
			TestUtilities.check("image BR pixel BR", gw + 49, gh + 24, g, Colors.BLACK);
			TestUtilities.check("image BR pixel BL", gw + 39, gh + 13, g, Colors.BLACK);

			TestUtilities.check("image BL pixel TL", gw + 28, gh - 11, g, Colors.RED);
			TestUtilities.check("image BL pixel TR", gw + 40, gh - 1, g, Colors.RED);
			TestUtilities.check("image BL pixel BR", gw + 28, gh + 12, g, Colors.BLACK);
			TestUtilities.check("image BL pixel BL", gw + 18, gh - 1, g, Colors.RED);
		}
	}

	/**
	 * Tests to clear a BVI after setting a null clip: in that case, the call to LLUI_DISPLAY_requestDrawing() returns
	 * false because nothing can be drawn in the graphics context.
	 */
	@Test
	public static void testClearNullClip() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(50, 50)) {

			GraphicsContext ig = image.getGraphicsContext();

			// fill the BVI in blue
			ig.setColor(Colors.BLUE);
			drawPath(ig);

			// set a null clip and clear the content of the BVI
			ig.setClip(0, 0, 0, 0);
			image.clear();

			// fill the TL part of the BVI in red
			ig.setClip(0, 0, (int) image.getWidth() / 2, (int) image.getHeight() / 2);
			ig.setColor(Colors.RED);
			drawPath(ig);

			// draw the BVI in the display
			VectorGraphicsPainter.drawImage(g, image, 0, 0);
			display.flush();

			// expected result: only the TL rectangle is drawn (in red)
			TestUtilities.check("pixel TL", 0, 0, g, Colors.RED);
			TestUtilities.check("pixel TR", 49, 0, g, TestUtilities.BACKGROUND_COLOR);
			TestUtilities.check("pixel BR", 49, 49, g, TestUtilities.BACKGROUND_COLOR);
			TestUtilities.check("pixel BL", 0, 49, g, TestUtilities.BACKGROUND_COLOR);
		}
	}

	private static void checkAreas(int x, int y, int w, int h) {
		checkAreas(x, y, w, h, Colors.RED);
	}

	private static void checkAreas(int x, int y, int w, int h, int c) {
		int errors = 0;
		errors += TestUtilities.checkPeripheralArea("clip over inside", c, x + 1, y + 1, w - 2, h - 2, 1, 0, false);
		errors += TestUtilities.checkPeripheralArea("clip over outside", TestUtilities.BACKGROUND_COLOR, x, y, w, h, 1,
				0, false);
		assertEquals(0, errors);
	}

	private static void drawPath(GraphicsContext g) {
		Path path = new Path();
		int w = g.getWidth();
		int h = g.getHeight();
		path.lineTo(w, 0);
		path.lineTo(w, h);
		path.lineTo(0, h);
		path.close();
		VectorGraphicsPainter.fillPath(g, path, 0, 0);
	}

	private static VectorImage getImage(int width, int height) {
		return VectorImage
				.getImage("/com/microej/microvg/test/test_buffered_vector_image_clip_" + width + "x" + height + ".xml");
	}
}
