/*
 * Java
 *
 * Copyright 2023-2025 MicroEJ Corp.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

/**
 * Metadata of firstfont.ttf.
 */
public class FirstFont {

	/**
	 * Value stored in the ttf (FontForge > Element > Font Info > General > Em Size). Unit of all sizes returned by this
	 * class.
	 */
	public static final int EM_SIZE = 1000;

	/**
	 * Character with right and left spaces.
	 */
	public static final FirstFontCharacter CENTERED = new CharacterCentered();

	/**
	 * Character with right space and no left space.
	 */
	public static final FirstFontCharacter LEFTLIMIT = new CharacterLeftLimit();

	/**
	 * Character with right space and a negative bearingX.
	 */
	public static final FirstFontCharacter LEFTOVER = new CharacterLeftOver();

	/**
	 * Character with left space and no right space.
	 */
	public static final FirstFontCharacter RIGHTLIMIT = new CharacterRightLimit();

	/**
	 * Character with left space and pixels over advanceX.
	 */
	public static final FirstFontCharacter RIGHTOVER = new CharacterRightOver();

	/**
	 * Combinaison of {@link #LEFTLIMIT} and {@link #RIGHTOVER}.
	 */
	public static final FirstFontCharacter LEFTLIMITRIGHTOVER = new CharacterLeftLimitRightOver();

	/**
	 * Combinaison of {@link #LEFTOVER} and {@link #RIGHTLIMIT}.
	 */
	public static final FirstFontCharacter LEFTOVERRIGHTLIMIT = new CharacterLeftOverRightLimit();

	/**
	 * Combinaison of {@link #LEFTOVER} and {@link #RIGHTOVER}.
	 */
	public static final FirstFontCharacter OVER = new CharacterOver();

	/**
	 * Character just before x0.
	 */
	public static final FirstFontCharacter OUTSIDELEFTLIMIT = new CharacterOutsideLeftLimit();

	/**
	 * Character before x0 and space between end of glyph and x0.
	 */
	public static final FirstFontCharacter OUTSIDELEFTOVER = new CharacterOutsideLeftOver();

	/**
	 * Character just after advanceX.
	 */
	public static final FirstFontCharacter OUTSIDERIGHTLIMIT = new CharacterOutsideRightLimit();

	/**
	 * Character after advanceX and space between advanceX and start of glyph.
	 */
	public static final FirstFontCharacter OUTSIDERIGHTOVER = new CharacterOutsideLeftOver();

	/**
	 * Null character (size == 0), just before x0.
	 */
	public static final FirstFontCharacter NULLLEFTLIMIT = new CharacterNullLeftLimit();

	/**
	 * Null character (size == 0), before x0 and space between end of glyph and x0.
	 */
	public static final FirstFontCharacter NULLLEFTOVER = new CharacterNullLeftOver();

	/**
	 * Null character (size == 0), just after advanceX.
	 */
	public static final FirstFontCharacter NULLRIGHTLIMIT = new CharacterNullRightLimit();

	/**
	 * Null character (size == 0), after advanceX and space between advanceX and start of glyph.
	 */
	public static final FirstFontCharacter NULLRIGHTOVER = new CharacterNullRightOver();

	/**
	 * Symbolizes a firstfont.ttf's character.
	 */
	public static abstract class FirstFontCharacter {

		/**
		 * Returns the character (each font's character is on one 16-bit word).
		 *
		 * @return the character
		 */
		public abstract char value();

		/**
		 * Returns the character as a string.
		 *
		 * @return the character
		 */
		public String string() {
			return String.valueOf(value());
		}

		/**
		 * Returns the space between the virtual vertical line at "x0" and first pixel drawn on the left. Returns 0 if
		 * the first drawn pixel on the left is before the virtual vertical line at "x0".
		 *
		 * @return a size based on {@link FirstFont#EM_SIZE}
		 */
		public abstract int leftSpace();

		/**
		 * Returns the width between the first pixel drawn on the left and the last pixel drawn on the right. The drawn
		 * glyph may start before the virtual vertical line at "x0" and end after the virtual vertical line at
		 * "advanceX".
		 *
		 * @return a size based on {@link FirstFont#EM_SIZE}
		 */
		public abstract int glyphWidth();

		/**
		 * Returns {@link #leftSpace()} when the first pixel drawn on the left is after the virtual vertical line at
		 * "x0", or returns a negative value otherwise (that means the glyph starts before the virtual vertical line at
		 * "x0").
		 *
		 * @return a size based on {@link FirstFont#EM_SIZE}
		 */
		public int bearingX() {
			// default value, overwrtten by specific characters
			return leftSpace();
		}

		/**
		 * Returns the size between the virtual vertical line "0" and the virtual vertical line at "advanceX".
		 *
		 * @return a size based on {@link FirstFont#EM_SIZE}
		 */
		public int advanceX() {
			// all characters use the same size
			return 1000;
		}

	}

	static class CharacterCentered extends FirstFontCharacter {

		@Override
		public char value() {
			return 'R';
		}

		@Override
		public int leftSpace() {
			return 100;
		}

		@Override
		public int glyphWidth() {
			return 500;
		}

	}

	static class CharacterLeftLimit extends FirstFontCharacter {

		@Override
		public char value() {
			return 'S';
		}

		@Override
		public int leftSpace() {
			return 0;
		}

		@Override
		public int glyphWidth() {
			return 800;
		}

	}

	static class CharacterLeftOver extends FirstFontCharacter {

		@Override
		public char value() {
			return 'M';
		}

		@Override
		public int leftSpace() {
			return 0;
		}

		@Override
		public int glyphWidth() {
			return 900;
		}

		@Override
		public int bearingX() {
			return -300;
		}
	}

	static class CharacterRightLimit extends FirstFontCharacter {

		@Override
		public char value() {
			return 'T';
		}

		@Override
		public int leftSpace() {
			return 100;
		}

		@Override
		public int glyphWidth() {
			return 900;
		}

	}

	static class CharacterRightOver extends FirstFontCharacter {

		@Override
		public char value() {
			return 'U';
		}

		@Override
		public int leftSpace() {
			return 100;
		}

		@Override
		public int glyphWidth() {
			return 1100;
		}
	}

	static class CharacterLeftLimitRightOver extends FirstFontCharacter {

		@Override
		public char value() {
			return 'X';
		}

		@Override
		public int leftSpace() {
			return 0;
		}

		@Override
		public int glyphWidth() {
			return 1200;
		}

	}

	static class CharacterLeftOverRightLimit extends FirstFontCharacter {

		@Override
		public char value() {
			return 'V';
		}

		@Override
		public int leftSpace() {
			return 0;
		}

		@Override
		public int bearingX() {
			return -300;
		}

		@Override
		public int glyphWidth() {
			return 1300;
		}

	}

	static class CharacterOver extends FirstFontCharacter {

		@Override
		public char value() {
			return 'W';
		}

		@Override
		public int leftSpace() {
			return 0;
		}

		@Override
		public int bearingX() {
			return -300;
		}

		@Override
		public int glyphWidth() {
			return 1400;
		}
	}

	static class CharacterOutsideLeftLimit extends FirstFontCharacter {

		@Override
		public char value() {
			return 'a';
		}

		@Override
		public int leftSpace() {
			return 0;
		}

		@Override
		public int bearingX() {
			return -400;
		}

		@Override
		public int glyphWidth() {
			return 400;
		}
	}

	static class CharacterOutsideLeftOver extends FirstFontCharacter {

		@Override
		public char value() {
			return 'b';
		}

		@Override
		public int leftSpace() {
			return 0;
		}

		@Override
		public int bearingX() {
			return -400;
		}

		@Override
		public int glyphWidth() {
			return 300;
		}
	}

	static class CharacterOutsideRightLimit extends FirstFontCharacter {

		@Override
		public char value() {
			return 'c';
		}

		@Override
		public int leftSpace() {
			return 1000;
		}

		@Override
		public int glyphWidth() {
			return 400;
		}

	}

	static class CharacterOutsideRightOver extends FirstFontCharacter {

		@Override
		public char value() {
			return 'd';
		}

		@Override
		public int leftSpace() {
			return 1100;
		}

		@Override
		public int glyphWidth() {
			return 300;
		}

	}

	static class CharacterNullLeftLimit extends FirstFontCharacter {

		@Override
		public char value() {
			return 'e';
		}

		@Override
		public int leftSpace() {
			return 0;
		}

		@Override
		public int bearingX() {
			return -400;
		}

		@Override
		public int glyphWidth() {
			return 400;
		}

		@Override
		public int advanceX() {
			return 0;
		}
	}

	static class CharacterNullLeftOver extends FirstFontCharacter {

		@Override
		public char value() {
			return 'f';
		}

		@Override
		public int leftSpace() {
			return 0;
		}

		@Override
		public int bearingX() {
			return -400;
		}

		@Override
		public int glyphWidth() {
			return 300;
		}

		@Override
		public int advanceX() {
			return 0;
		}
	}

	static class CharacterNullRightLimit extends FirstFontCharacter {

		@Override
		public char value() {
			return 'g';
		}

		@Override
		public int leftSpace() {
			return 0;
		}

		@Override
		public int glyphWidth() {
			return 300;
		}

		@Override
		public int advanceX() {
			return 0;
		}
	}

	static class CharacterNullRightOver extends FirstFontCharacter {

		@Override
		public char value() {
			return 'h';
		}

		@Override
		public int leftSpace() {
			return 100;
		}

		@Override
		public int glyphWidth() {
			return 200;
		}

		@Override
		public int advanceX() {
			return 0;
		}
	}
}
