/*
 * Java
 *
 * Copyright 2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.control.net;

import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.Socket;

/**
 * Socket output stream wrapper used to control net/ssl write call.
 */
public class SocketOutputStream extends FilterOutputStream {

	private final Socket socket;
	private static final Object lock = new Object();

	/**
	 * Creates a socket output stream instance.
	 *
	 * @param socket
	 * 		the socket.
	 * @param outputstream
	 * 		the underlying output stream.
	 * @throws IOException
	 * 		if an error occurs.
	 */
	public SocketOutputStream(Socket socket, OutputStream outputstream) throws IOException {
		super(outputstream);
		this.socket = socket;
	}

	@Override
	public void write(int b) throws IOException {
		synchronized (lock) {
			WriteSocket writeSocket = new WriteSocket(this.socket, 1);
			writeSocket.onStart();
			try {
				super.write(b);
				writeSocket.onEnd(true);
			} catch (IOException | IllegalArgumentException | SecurityException e) {
				writeSocket.onEnd(false);
				throw e;
			}
		}
	}

	@Override
	public void write(byte[] b, int off, int len) throws IOException {
		synchronized (lock) {
			WriteSocket writeSocket = new WriteSocket(this.socket, len);
			writeSocket.onStart();
			try {
				this.out.write(b, off, len);
				writeSocket.onEnd(true);
			} catch (IOException | IllegalArgumentException | SecurityException e) {
				writeSocket.onEnd(false);
				throw e;
			}
		}
	}

	@Override
	public void close() throws IOException {
		synchronized (lock) {
			boolean wasClosed = this.socket.isClosed();
			super.close();
			if (!wasClosed) {
				CloseSocket closeSocket = new CloseSocket(this.socket);
				closeSocket.onEnd(true);
			}
		}
	}
}
