/*
 * Java
 *
 * Copyright 2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microvg.image;

import java.nio.ByteBuffer;

import ej.microvg.CommandDecoder;
import ej.microvg.LLVGConstants;
import ej.microvg.image.pathdata.PathData;
import ej.microvg.paint.LinearGradient;

/**
 * Represents the decoder able to decode the images encoded for a given GPU.
 */
public interface ImageDecoder extends CommandDecoder {

	/**
	 * Wraps a {@link PathData} able to decode the path encoded in the RAW image's buffer. The first path element (the
	 * path's header) starts at current byte buffer's position.
	 *
	 * To decode the path's data, the implementation should wrap another {@link ByteBuffer}:
	 *
	 * <pre>
	 * ByteBuffer buffer = ByteBuffer.wrap(is.array(), is.position(), length).order(ByteOrder.LITTLE_ENDIAN);
	 * </pre>
	 *
	 * @param buffer
	 *            the buffer where is encoded the path
	 * @return a path buffer
	 */
	PathData decodePath(ByteBuffer buffer);

	/**
	 * Decodes the linear gradient encoded in the RAW image's buffer. The first gradient element (the gradient's header)
	 * starts at current byte buffer's position.
	 *
	 * @param buffer
	 *            the buffer where is encoded the gradient
	 * @return a {@link LinearGradient}
	 * @see #decodePath(ByteBuffer)
	 */
	LinearGradient decodeGradient(ByteBuffer buffer);

	/**
	 * Decodes the fill rule encoded for a GPU.
	 *
	 * @param rule
	 *            the fill rule to decode
	 * @return a {@link LLVGConstants} fill rule
	 */
	int decodeFillRule(int rule);

	/**
	 * Decodes the color encoded for a GPU. The expected color is a 32-bit color A-R-G-B.
	 *
	 * Default implementation considers the color is already a ARGB 32-bit color and does nothing.
	 *
	 * @param color
	 *            color the color to decode
	 * @return a ARGB 32-bit color
	 */
	default int decodeColor(int color) {
		return color;
	}
}
