/*
 * Java
 *
 * Copyright 2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.hoka.log;

import ej.hoka.http.Config;

/**
 *
 */
class SimpleLogger implements Logger {

	private static final int TRACE = 0x11111;
	private static final int DEBUG = 0x01111;
	private static final int INFO = 0x00111;
	private static final int WARN = 0x00011;
	private static final int ERROR = 0x00001;

	private final int level;

	/**
	 *
	 */
	public SimpleLogger() {
		String slevel = Config.getInstance().getLoggerLevel();
		if (slevel != null && !slevel.isEmpty()) {
			switch (slevel.trim().toLowerCase()) {
			case "none": //$NON-NLS-1$
				this.level = 0x00000;
				break;
			case "trace": //$NON-NLS-1$
				this.level = TRACE;
				break;
			case "debug": //$NON-NLS-1$
				this.level = DEBUG;
				break;
			case "info": //$NON-NLS-1$
				this.level = INFO;
				break;
			case "warn": //$NON-NLS-1$
				this.level = WARN;
				break;
			case "error": //$NON-NLS-1$
				this.level = ERROR;
				break;
			default:
				this.level = INFO;
				break;
			}
		} else {
			this.level = INFO;
		}
	}

	@Override
	public void trace(String msg) {
		if (isTraceEnabled()) {
			System.out.println("[TRACE]\t" + msg); //$NON-NLS-1$
		}
	}

	@Override
	public void debug(String msg) {
		if (isDebugEnabled()) {
			System.out.println("[DEBUG]\t" + msg); //$NON-NLS-1$
		}
	}

	@Override
	public void info(String msg) {
		if (isInfoEnabled()) {
			System.out.println("[INFO]\t" + msg); //$NON-NLS-1$
		}
	}

	@Override
	public void warn(String msg) {
		if (isWarnEnabled()) {
			System.out.println("[WARN]\t" + msg); //$NON-NLS-1$
		}
	}

	@Override
	public void error(String msg) {
		if (isErrorEnabled()) {
			System.out.println("[ERROR]\t" + msg); //$NON-NLS-1$
		}
	}

	@Override
	public void error(String msg, Throwable e) {
		if (isErrorEnabled()) {
			System.out.println("[ERROR]\t" + msg); //$NON-NLS-1$
			e.printStackTrace();
		}

	}

	@Override
	public void error(Throwable e) {
		if (isErrorEnabled()) {
			System.out.println("[ERROR]\t" + e.getMessage()); //$NON-NLS-1$
			e.printStackTrace();
		}
	}

	@Override
	public boolean isTraceEnabled() {
		return (TRACE & this.level) == TRACE;
	}

	@Override
	public boolean isDebugEnabled() {
		return (DEBUG & this.level) == DEBUG;
	}

	@Override
	public boolean isInfoEnabled() {
		return (INFO & this.level) == INFO;
	}

	@Override
	public boolean isWarnEnabled() {
		return (WARN & this.level) == WARN;
	}

	@Override
	public boolean isErrorEnabled() {
		return (ERROR & this.level) == ERROR;
	}

}
