/*
 * Copyright 2023 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */

package ej.widget.event;

import ej.annotation.Nullable;
import ej.mwt.Widget;
import ej.widget.basic.OnClickListener;

/**
 * The click event handler responsibility is to detect pointer events (press, drag, release) and help interact with a
 * {@link Clickable button-like interactive} element.
 *
 * @since 4.3.0
 */
public class ClickEventHandler extends PointerEventHandler {

	private final Clickable clickable;
	private @Nullable OnClickListener onClickListener;

	/**
	 * Creates a swipe event handler on an element.
	 *
	 * @param widget
	 *            the attached widget.
	 * @param clickable
	 *            the clickable.
	 */
	public ClickEventHandler(Widget widget, Clickable clickable) {
		super(widget);
		this.clickable = clickable;
	}

	/**
	 * Sets the "on click" listener.
	 *
	 * @param onClickListener
	 *            the "on click" listener to set or <code>null</code>.
	 */
	public void setOnClickListener(@Nullable OnClickListener onClickListener) {
		this.onClickListener = onClickListener;
	}

	@Override
	protected boolean onPressed(int pointerX, int pointerY) {
		this.clickable.setPressed(true);
		return false;
	}

	@Override
	protected boolean onReleased(int pointerX, int pointerY) {
		this.clickable.setPressed(false);
		handleClick();
		return true;
	}

	@Override
	protected void onExited() {
		this.clickable.setPressed(false);
	}

	/**
	 * Handles a click event.
	 *
	 * @deprecated Internal API.
	 */
	@Deprecated
	public void handleClick() {
		OnClickListener listener = this.onClickListener;
		if (listener != null) {
			listener.onClick();
		}
	}

}
