/*
 * Copyright 2013-2023 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.widget.color;

import ej.microui.display.Colors;

/**
 * Color utilities regarding lightness.
 */
public class LightHelper {

	private static final int LIGHT_COLOR_LIMIT = 0x7f;
	private static final float BLUE_LIGHT_FACTOR = 0.114f;
	private static final float GREEN_LIGHT_FACTOR = 0.587f;
	private static final float RED_LIGHT_FACTOR = 0.299f;
	private static final int LIGHT_FACTOR = 0x10;

	// Prevents initialization.
	private LightHelper() {
	}

	/**
	 * Gets the darkest of two colors.
	 *
	 * @param color1
	 *            first color.
	 * @param color2
	 *            second color.
	 * @return color1 if its light level is lower than the one of color2, color2 otherwise.
	 */
	public static int getDarkestColor(int color1, int color2) {
		int result = color1;
		if (getLight(color2) < getLight(color1)) {
			result = color2;
		}
		return result;
	}

	/**
	 * Gets the lightest of two colors.
	 *
	 * @param color1
	 *            first color.
	 * @param color2
	 *            second color.
	 * @return color1 if its light level is higher than the one of color2, color2 otherwise.
	 */
	public static int getLightestColor(int color1, int color2) {
		int result = color1;
		if (getLight(color2) > getLight(color1)) {
			result = color2;
		}
		return result;
	}

	/**
	 * Gets a darkened version of a color.
	 * <p>
	 * Note that passing a negative factor value, the method will return a lighter color.
	 *
	 * @param color
	 *            the color to darken.
	 * @param factor
	 *            the darkening factor.
	 * @return the darkened color.
	 */
	public static int darkenColor(int color, int factor) {
		return modifyColorLight(color, -factor);
	}

	/**
	 * Gets a lightened version of a color.
	 * <p>
	 * Note that passing a negative factor value, the method will return a darker color.
	 *
	 * @param color
	 *            the color to lighten.
	 * @param factor
	 *            the lightening factor.
	 * @return the lightened color.
	 */
	public static int lightenColor(int color, int factor) {
		return modifyColorLight(color, factor);
	}

	private static int modifyColorLight(int color, int factor) {
		int red = ColorHelper.getRed(color);
		int green = ColorHelper.getGreen(color);
		int blue = ColorHelper.getBlue(color);

		factor = factor * LIGHT_FACTOR;

		red = ColorHelper.updateComponent(red, factor);
		green = ColorHelper.updateComponent(green, factor);
		blue = ColorHelper.updateComponent(blue, factor);

		return ColorHelper.getColor(red, green, blue);
	}

	/**
	 * Gets whether a color is light or dark.
	 *
	 * @param color
	 *            the color to test.
	 * @return {@code true} if the color is a light one, {@code false} otherwise.
	 */
	public static boolean isLightColor(int color) {
		int light = getLight(color);
		return light > LIGHT_COLOR_LIMIT;
	}

	/**
	 * Gets the light level of a color.
	 *
	 * @param color
	 *            the color.
	 * @return the light level of the given color.
	 */
	public static int getLight(int color) {
		int red = ColorHelper.getRed(color);
		int green = ColorHelper.getGreen(color);
		int blue = ColorHelper.getBlue(color);
		return (int) (red * RED_LIGHT_FACTOR + green * GREEN_LIGHT_FACTOR + blue * BLUE_LIGHT_FACTOR);
	}

	/**
	 * Gets the color that best contrasts against a given color. Result is either {@link Colors#WHITE} or
	 * {@link Colors#BLACK}.
	 *
	 * @param colorToContrast
	 *            the color to contrast.
	 * @return {@link Colors#BLACK} if the given color is light, {@link Colors#WHITE} otherwise.
	 * @see #isLightColor(int)
	 */
	public static int getMostContrastingColor(int colorToContrast) {
		return getMostContrastingColor(colorToContrast, Colors.WHITE, Colors.BLACK);
	}

	/**
	 * Gets the color that best contrasts against a given color.
	 *
	 * @param colorToContrast
	 *            the color to contrast.
	 * @param color1
	 *            the first color.
	 * @param color2
	 *            the second color.
	 * @return the darkest color of {first color,second color} if the given color is light, the lightest color
	 *         otherwise.
	 * @see #isLightColor(int)
	 */
	public static int getMostContrastingColor(int colorToContrast, int color1, int color2) {
		if (isLightColor(colorToContrast)) {
			return getDarkestColor(color1, color2);
		} else {
			return getLightestColor(color1, color2);
		}
	}

}
