/*
 * Copyright 2014-2017 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.motion.util;

import java.util.List;

import ej.bon.Timer;
import ej.bon.TimerTask;
import ej.motion.Motion;

/**
 * Utility class to schedule a {@link Motion} on a {@link Timer}.
 */
public class MotionAnimator {

	private final List<Motion> motions;
	private final boolean loop;
	private final MotionListener listener;

	private Motion currentMotion;
	private int currentMotionIndex;

	private TimerTask task;

	/**
	 * Creates a motion animator with the specified motion.
	 * <p>
	 * The given motion listener will be notified:
	 * <ul>
	 * <li>at startup (see {@link #start(Timer, long)},</li>
	 * <li>for each new value (see {@link Motion#getCurrentValue()},</li>
	 * <li>when the motion is finished (see {@link Motion#isFinished()},</li>
	 * <li>if the animator is stopped (see {@link #stop()}.</li>
	 * </ul>
	 *
	 * @param motion
	 *            the motion to animate.
	 * @param listener
	 *            the listener to notify for each step.
	 */
	public MotionAnimator(Motion motion, MotionListener listener) {
		this(null, motion, listener, false);
	}

	/**
	 * Creates a motion animator with the specified motion.
	 * <p>
	 * The given motion listener will be notified:
	 * <ul>
	 * <li>at startup (see {@link #start(Timer, long)},</li>
	 * <li>for each new value (see {@link Motion#getCurrentValue()},</li>
	 * <li>when the motion is finished (see {@link Motion#isFinished()},</li>
	 * <li>if the animator is stopped (see {@link #stop()}.</li>
	 * </ul>
	 *
	 * @param motions
	 *            the list of motions to animate.
	 * @param listener
	 *            the listener to notify for each step.
	 * @param loop
	 *            if true, then the motion should loop until stopped.
	 * @throws IndexOutOfBoundsException
	 *             if there is no item in the motions list.
	 */
	public MotionAnimator(List<Motion> motions, MotionListener listener, boolean loop) {
		this(motions, motions.get(0), listener, loop);
	}

	private MotionAnimator(List<Motion> motions, Motion motion, MotionListener listener, boolean loop) {
		this.motions = motions;
		this.listener = listener;
		this.currentMotion = motion;
		this.loop = loop;
	}

	/**
	 * Starts the animation on the given timer.
	 * <p>
	 * Calls {@link MotionListener#start(int)} on the registered listener with the start value of the motion.
	 * <p>
	 * If the motion animator is already started, it is firstly stopped.
	 *
	 * @param timer
	 *            the timer to schedule on.
	 * @param period
	 *            time in milliseconds between motion steps.
	 *
	 * @see Timer#schedule(TimerTask, long, long)
	 * @see Motion#getStartValue()
	 */
	public void start(Timer timer, long period) {
		stop();
		this.currentMotion.start();
		this.listener.start(this.currentMotion.getStartValue());
		MotionTask task = new MotionTask();
		synchronized (this) {
			timer.schedule(task, 0, period);
			this.task = task;
		}
	}

	/**
	 * Stops the animation.
	 * <p>
	 * Calls {@link MotionListener#stop(int)} on the registered listener with the stop value of the motion.
	 *
	 * @see Motion#getStopValue()
	 */
	public void stop() {
		if (cancelTask()) {
			this.listener.stop(this.currentMotion.getStopValue());
		}
	}

	/**
	 * Cancels the animation.
	 * <p>
	 * Calls {@link MotionListener#cancel()} on the registered listener.
	 */
	public void cancel() {
		if (cancelTask()) {
			this.listener.cancel();
		}
	}

	/**
	 * Cancels the task if running.
	 *
	 * @return {@code true} if the task was running, {@code false} otherwise.
	 */
	private boolean cancelTask() {
		TimerTask oldTask;
		synchronized (this) {
			oldTask = this.task;
			if (oldTask == null) {
				return false;
			}
			this.task = null;
		}
		oldTask.cancel();
		return true;
	}

	private class MotionTask extends TimerTask {

		/**
		 * Gets the current value of the motion and notifies the listener by calling {@link MotionListener#step(int)}
		 * passing this value.
		 * <p>
		 * If the motion is finished, notifies the listener by calling {@link MotionListener#stop(int)} passing the stop
		 * value of the motion.
		 *
		 * @see Motion#getCurrentValue()
		 * @see Motion#isFinished()
		 * @see Motion#getCurrentValue()
		 * @see MotionListener#step(int)
		 */
		@Override
		public void run() {
			// System.out.println("MotionAnimator.run()");
			if (MotionAnimator.this.currentMotion.isFinished()) {
				List<Motion> motionsLocal = MotionAnimator.this.motions;
				if (motionsLocal == null) {
					stop();
					return;
				} else {
					if (motionsLocal.size() == ++MotionAnimator.this.currentMotionIndex) {
						if (MotionAnimator.this.loop) {
							// restart
							MotionAnimator.this.currentMotionIndex = 0;
						} else {
							stop();
							return;
						}
					}
					MotionAnimator.this.currentMotion = motionsLocal.get(MotionAnimator.this.currentMotionIndex);
					// System.out.println("-> " + currentMotionIndex + " " + currentMotion);
					MotionAnimator.this.currentMotion.start();
				}
			}
			int value = MotionAnimator.this.currentMotion.getCurrentValue();
			// System.out.println("MotionAnimator.run() " + value);
			MotionAnimator.this.listener.step(value);
		}

	}

}
