/*
 * Java
 *
 * Copyright 2020-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

/**
 * This class provides implementations of native methods of MicroUI library which perform some drawings.
 * <p>
 * The equivalent of embedded structures and enumerations are {@link MicroUIImage}, {@link MicroUIGraphicsContext} and
 * {@link MicroUIImageFormat}. They are used to identify the MicroUI graphical objects: GraphicsContext, Image, image
 * format. etc.
 * <p>
 * Implementation must use the framework object {@link LLUIDisplay#Instance} as monitor to synchronize drawings.
 * <p>
 * All native methods have a byte array as the first parameter. This identifies the destination target: the MicroUI
 * GraphicsContext. This target is retrieved in MicroEJ application calling the method <code>gc.getSNIContext()</code>.
 * The returned byte array must be mapped on a {@link MicroUIGraphicsContext} object in MicroUI native drawing method
 * implementation (see {@link LLUIDisplay#mapMicroUIGraphicsContext(byte[])}).
 * <p>
 * For a MicroUI Image, MicroEJ application has to call <code>image.getSNIContext()</code> to get the image byte array.
 * This byte array must be mapped it on a {@link MicroUIImage} object in MicroUI native drawing method implementation
 * (see {@link LLUIDisplay#mapMicroUIImage(byte[])}).
 * <p>
 * A graphics context holds a clip and the drawer is not allowed to perform a drawing outside this clip (otherwise the
 * behavior is unknown). Note the bottom-right coordinates might be smaller than top-left (in x and/or y) when the clip
 * width and/or height is null. The clip may be disabled (when the current drawing fits the clip); this allows to reduce
 * runtime. See {@link MicroUIGraphicsContext#isClipEnabled()}. Note: several clip methods are available in
 * {@link MicroUIGraphicsContext} to check if a drawing fits the clip.
 * <p>
 * The drawing method must:
 * <ul>
 * <li>Convert the MicroUI graphics context color calling {@link MicroUIGraphicsContext#getRenderingColor()} (in order
 * to respect embedded display pixel representation constraints),</li>
 * <li>Respect the graphics context clip; if not respected, the front panel behavior is unknown.</li>
 * </ul>
 *
 * <p>
 * The native method implementation pattern is:
 *
 * <pre>
 * 	void drawXxx(byte[] target, ...) {
 *
 * 		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
 *
 * 		// all calls to GraphicsEngine.Instance APIs must be synchronized on GraphicsEngine.Instance itself
 * 		synchronized (graphicsEngine) {
 *
 * 			// map a MicroUIGraphicsContext on given byte array which represents the destination target
 * 			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
 *
 * 			// ask to the Graphics Engine if a drawing can be performed on target
 * 			if (gc.requestDrawing()) {
 *
 *				// call current drawer's equivalent drawing method
 *				LLUIDisplay.Instance.getUIDrawer(gc).drawXxx(gc, x, y);
 *
 *				// contrary to MicroUI native drawing methods implementation for embedded (LLUI_PAINTER_impl.h), there
 *				// is no status to set in MicroUI Graphics Engine
 *			}
 *		}
 *	}
 * </pre>
 */
public class LLUIPainter {

	/**
	 * Value used when no incident occurred.
	 */
	public static final int DRAWING_SUCCESS = 0;

	/**
	 * Flag stating that an error occurred during a drawing.
	 *
	 * This flag will cause an exception to be thrown when checking the flags in the application. No exception will be
	 * thrown if this flag is not set, although other flags will keep their state and be readable in the application.
	 * This flag is to be combined with other flags describing the error.
	 */
	public static final int DRAWING_LOG_ERROR = 1 << 31;

	/**
	 * Flag stating that a drawing function is lacking an implementation.
	 *
	 * This flag is typically set when the format of the graphics context is not supported by the implementation.
	 */
	public static final int DRAWING_LOG_NOT_IMPLEMENTED = 1 << 0;

	/**
	 * Flag stating that a function was called in a context that does not allow that operation.
	 *
	 * This flag is typically set when a drawing function has been disabled for a specific format, like a vector drawing
	 * function called on a raster-only graphics context.
	 */
	public static final int DRAWING_LOG_FORBIDDEN = 1 << 1;

	/**
	 * Flag stating that the system ran out of memory while attempting to perform a drawing.
	 */
	public static final int DRAWING_LOG_OUT_OF_MEMORY = 1 << 2;

	/**
	 * Flag stating that the clip area of a graphics context was modified by
	 * {@link MicroUIGraphicsContext#setClip(int, int, int, int)} or
	 * {@link MicroUIGraphicsContext#intersectClip(int, int, int, int)}.
	 *
	 * If this flag is set, the caller is responsible for saving the previous clip area and restore it afterwards.
	 *
	 * This flag merely warns the user that the clip values returned by <code>GraphicsContext.getClipX</code>,
	 * <code>GraphicsContext.getClipY</code>, <code>GraphicsContext.getClipWidth</code> and
	 * <code>GraphicsContext.getClipHeight</code> may not be identical to the clip values used in the low-level side. It
	 * is meant to be used as a debugging hint if a drawing seems incorrect.
	 *
	 * @see MicroUIGraphicsContext#setClip(int, int, int, int)
	 * @see MicroUIGraphicsContext#intersectClip(int, int, int, int)
	 */
	public static final int DRAWING_LOG_CLIP_MODIFIED = 1 << 3;

	/**
	 * Flag stating that an undefined character was drawn.
	 *
	 * This happens when drawing a string that contains a character that is not included in the font used.
	 */
	public static final int DRAWING_LOG_MISSING_CHARACTER = 1 << 4;

	/**
	 * Flag describing incidents occurring in a drawing library.
	 *
	 * Refer to the MicroUI implementation in the VEE port for more information about this incident.
	 */
	public static final int DRAWING_LOG_LIBRARY_INCIDENT = 1 << 29;

	/**
	 * Flag describing incidents that do not match other values.
	 */
	public static final int DRAWING_LOG_UNKNOWN_INCIDENT = 1 << 30;

	private LLUIPainter() {
		// forbidden constructor: there is no instance of LLUIPainter
	}

	/**
	 * MicroUI native "writePixel". Draws a pixel at given position.
	 * <p>
	 * See class comment to have more information about native method implementation.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param x
	 *            the pixel X coordinate.
	 * @param y
	 *            the pixel Y coordinate.
	 */
	public static void writePixel(byte[] target, int x, int y) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && gc.isPointInClip(x, y)) {
				gc.configureClip(false /* point is in clip */);
				LLUIDisplay.Instance.getUIDrawer(gc).writePixel(gc, x, y);
			}
		}
	}

	/**
	 * MicroUI native "drawLine". Draws a line at between points x1,y1 (included) and x2,y2 (included). Note: x1 may be
	 * higher than x2 (and/or y1 may be higher than y2).
	 * <p>
	 * See class comment to have more information about native method implementation.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param x1
	 *            the first pixel line X coordinate.
	 * @param y1
	 *            the first pixel line Y coordinate.
	 * @param x2
	 *            the last pixel line X coordinate.
	 * @param y2
	 *            the last pixel line Y coordinate.
	 */
	public static void drawLine(byte[] target, int x1, int y1, int x2, int y2) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing()) {
				// cannot reduce/clip line: may be x2 < x1 and / or y2 < y1
				LLUIDisplay.Instance.getUIDrawer(gc).drawLine(gc, x1, y1, x2, y2);
			}
		}
	}

	/**
	 * MicroUI native "drawHorizontalLine". Draws a horizontal line at between points x,y (included) and x+width-1,y
	 * (included).
	 * <p>
	 * See class comment to have more information about native method implementation.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param x
	 *            the first pixel line X coordinate.
	 * @param y
	 *            the first pixel line Y coordinate.
	 * @param width
	 *            the line size.
	 */
	public static void drawHorizontalLine(byte[] target, int x, int y, int width) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && width > 0) {
				int x1 = x;
				int x2 = x + width - 1;
				// can disable clip when line is fully in clip
				gc.configureClip(!gc.isHorizontalLineInClip(x1, x2, y));
				LLUIDisplay.Instance.getUIDrawer(gc).drawHorizontalLine(gc, x1, x2, y);
			}
		}
	}

	/**
	 * MicroUI native "drawVerticalLine". Draws a vertical line at between points x,y (included) and x,y+height-1
	 * (included).
	 * <p>
	 * See class comment to have more information about native method implementation.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param x
	 *            the first pixel line X coordinate.
	 * @param y
	 *            the first pixel line Y coordinate.
	 * @param height
	 *            the line size.
	 */
	public static void drawVerticalLine(byte[] target, int x, int y, int height) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && height > 0) {
				int y1 = y;
				int y2 = y + height - 1;
				// can disable clip when line is fully in clip
				gc.configureClip(!gc.isVerticalLineInClip(y1, y2, x));
				LLUIDisplay.Instance.getUIDrawer(gc).drawVerticalLine(gc, x, y1, y2);
			}
		}
	}

	/**
	 * MicroUI native "drawRectangle". Draws a rectangle at from top-left point x,y (included) and bottom-right point
	 * x+width-1,y+height-1 (included).
	 * <p>
	 * See class comment to have more information about native method implementation.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the rectangle width.
	 * @param height
	 *            the rectangle height.
	 */
	public static void drawRectangle(byte[] target, int x, int y, int width, int height) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && width > 0 && height > 0) {
				int x1 = x;
				int x2 = x + width - 1;
				int y1 = y;
				int y2 = y + height - 1;
				// can disable clip when rectangle is fully in clip
				gc.configureClip(!gc.isRectangleInClip(x1, y1, x2, y2));
				LLUIDisplay.Instance.getUIDrawer(gc).drawRectangle(gc, x1, y1, x2, y2);
			}
		}
	}

	/**
	 * MicroUI native "fillRectangle". Fills a rectangle at from top-left point x1,y1 (included) and bottom-right point
	 * x2,y2 (included). Caller ensures that x1 <= x2 and y1 <= y2.
	 * <p>
	 * See class comment to have more information about native method implementation.
	 *
	 * @param target
	 *            drawing destination target: a byte array to map on a {@link MicroUIGraphicsContext}.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the rectangle width.
	 * @param height
	 *            the rectangle height.
	 */
	public static void fillRectangle(byte[] target, int x, int y, int width, int height) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && width > 0 && height > 0) {
				int x1 = x;
				int x2 = x + width - 1;
				int y1 = y;
				int y2 = y + height - 1;
				// can disable clip when rectangle is fully in clip
				gc.configureClip(!gc.isRectangleInClip(x1, y1, x2, y2));
				LLUIDisplay.Instance.getUIDrawer(gc).fillRectangle(gc, x1, y1, x2, y2);
			}
		}
	}

	/**
	 * MicroUI native "drawRoundedRectangle". Draws a rounded rectangle at from top-left point x,y (included) and
	 * bottom-right point x+width-1,y+height-1 (included).
	 *
	 * @param target
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the rectangle width.
	 * @param height
	 *            the rectangle height.
	 * @param arcWidth
	 *            the horizontal diameter of the arc at the corners.
	 * @param arcHeight
	 *            the vertical diameter of the arc at the corners.
	 */
	public static void drawRoundedRectangle(byte[] target, int x, int y, int width, int height, int arcWidth,
			int arcHeight) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && width > 0 && height > 0) {
				// can disable clip when rectangle is fully in clip
				gc.configureClip(!gc.isRegionInClip(x, y, width, height));
				LLUIDisplay.Instance.getUIDrawer(gc).drawRoundedRectangle(gc, x, y, width, height, arcWidth, arcHeight);
			}
		}
	}

	/**
	 * MicroUI native "fillRoundedRectangle". Fills a rounded rectangle at from top-left point x,y (included) and
	 * bottom-right point x+width-1,y+height-1 (included).
	 *
	 * @param target
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the rectangle width.
	 * @param height
	 *            the rectangle height.
	 * @param arcWidth
	 *            the horizontal diameter of the arc at the corners.
	 * @param arcHeight
	 *            the vertical diameter of the arc at the corners.
	 */
	public static void fillRoundedRectangle(byte[] target, int x, int y, int width, int height, int arcWidth,
			int arcHeight) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && width > 0 && height > 0) {
				// can disable clip when rectangle is fully in clip
				gc.configureClip(!gc.isRegionInClip(x, y, width, height));
				LLUIDisplay.Instance.getUIDrawer(gc).fillRoundedRectangle(gc, x, y, width, height, arcWidth, arcHeight);
			}
		}
	}

	/**
	 * MicroUI native "drawCircleArc". Draws a circular arc covering the square defined by top-left point x,y (included)
	 * and bottom-right point x+diameter-1,y+diameter-1 (included).
	 * <p>
	 * The arc is drawn from startAngle up to arcAngle degrees. The center of the arc is defined as the center of the
	 * square whose origin is at (x,y) (upper-left corner) and whose dimension is given by diameter.
	 * <p>
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 * <p>
	 * If either the given diameter is negative or zero, or if arcAngle is zero, nothing is drawn.
	 * <p>
	 * The angles are given relative to the square. For instance an angle of 45 degrees is always defined by the line
	 * from the center of the square to the upper right corner of the square.
	 *
	 * @param target
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param diameter
	 *            the diameter of the arc to draw.
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from startAngle
	 */
	public static void drawCircleArc(byte[] target, int x, int y, int diameter, float startAngle, float arcAngle) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && diameter > 0 && arcAngle != 0) {
				// can disable clip when arc is fully in clip
				gc.configureClip(!gc.isRegionInClip(x, y, diameter, diameter));
				LLUIDisplay.Instance.getUIDrawer(gc).drawCircleArc(gc, x, y, diameter, startAngle, arcAngle);
			}
		}
	}

	/**
	 * MicroUI native "drawEllipseArc". Draws an elliptical arc covering the rectangle defined by top-left point x,y
	 * (included) and bottom-right point x+width-1,y+height-1 (included).
	 * <p>
	 * The arc is drawn from startAngle up to arcAngle degrees. The center of the arc is defined as the center of the
	 * rectangle whose origin is at (x,y) (upper-left corner) and whose dimension is given by width and height.
	 * <p>
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 * <p>
	 * If either the given width or height is negative or zero, or if arcAngle is zero, nothing is drawn.
	 * <p>
	 * The angles are given relative to the rectangle. For instance an angle of 45 degrees is always defined by the line
	 * from the center of the rectangle to the upper right corner of the rectangle. Thus for a non squared rectangle
	 * angles are skewed along either height or width.
	 *
	 * @param target
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the rectangle width.
	 * @param height
	 *            the rectangle height.
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from startAngle
	 */
	public static void drawEllipseArc(byte[] target, int x, int y, int width, int height, float startAngle,
			float arcAngle) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && width > 0 && height > 0 && arcAngle != 0) {
				// can disable clip when arc is fully in clip
				gc.configureClip(!gc.isRegionInClip(x, y, width, height));
				LLUIDisplay.Instance.getUIDrawer(gc).drawEllipseArc(gc, x, y, width, height, startAngle, arcAngle);
			}
		}
	}

	/**
	 * MicroUI native "fillCircleArc". Fills a circular arc covering the square defined by top-left point x,y (included)
	 * and bottom-right point x+diameter-1,y+diameter-1 (included).
	 *
	 * The arc is drawn from startAngle up to arcAngle degrees. The center of the arc is defined as the center of the
	 * square whose origin is at (x,y) (upper-left corner) and whose dimension is given by width and height.
	 *
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 *
	 * If either the given diameter is negative or zero, or if arcAngle is zero, nothing is drawn.
	 *
	 * The angles are given relative to the square. For instance an angle of 45 degrees is always defined by the line
	 * from the center of the square to the upper right corner of the square.
	 *
	 * @param target
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param diameter
	 *            the diameter of the arc to draw.
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from startAngle
	 */
	public static void fillCircleArc(byte[] target, int x, int y, int diameter, float startAngle, float arcAngle) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && diameter > 0 && arcAngle != 0) {
				// can disable clip when arc is fully in clip
				gc.configureClip(!gc.isRegionInClip(x, y, diameter, diameter));
				LLUIDisplay.Instance.getUIDrawer(gc).fillCircleArc(gc, x, y, diameter, startAngle, arcAngle);
			}
		}
	}

	/**
	 * MicroUI native "fillEllipseArc". Fills an arc covering the rectangle defined by top-left point x,y (included) and
	 * bottom-right point x+width-1,y+height-1 (included).
	 * <p>
	 * The arc is drawn from startAngle up to arcAngle degrees. The center of the arc is defined as the center of the
	 * rectangle whose origin is at (x,y) (upper-left corner) and whose dimension is given by width and height.
	 * <p>
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 * <p>
	 * If either the given width or height is negative or zero, or if arcAngle is zero, nothing is drawn.
	 * <p>
	 * The angles are given relative to the rectangle. For instance an angle of 45 degrees is always defined by the line
	 * from the center of the rectangle to the upper right corner of the rectangle. Thus for a non squared rectangle
	 * angles are skewed along either height or width.
	 *
	 * @param target
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the rectangle width.
	 * @param height
	 *            the rectangle height.
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from startAngle
	 */
	public static void fillEllipseArc(byte[] target, int x, int y, int width, int height, float startAngle,
			float arcAngle) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && width > 0 && height > 0 && arcAngle != 0) {
				// can disable clip when arc is fully in clip
				gc.configureClip(!gc.isRegionInClip(x, y, width, height));
				LLUIDisplay.Instance.getUIDrawer(gc).fillEllipseArc(gc, x, y, width, height, startAngle, arcAngle);
			}
		}
	}

	/**
	 * MicroUI native "drawEllipse". Draws an ellipse covering the rectangle defined by top-left point x,y (included)
	 * and bottom-right point x+width-1,y+height-1 (included).
	 * <p>
	 * If either the given width or height is negative or zero, nothing is drawn.
	 *
	 * @param target
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the ellipse width.
	 * @param height
	 *            the ellipse height.
	 */

	public static void drawEllipse(byte[] target, int x, int y, int width, int height) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && width > 0 && height > 0) {
				// can disable clip when ellipse is fully in clip
				gc.configureClip(!gc.isRegionInClip(x, y, width, height));
				LLUIDisplay.Instance.getUIDrawer(gc).drawEllipse(gc, x, y, width, height);

			}
		}
	}

	/**
	 * MicroUI native "fillEllipse". Fills an ellipse covering the rectangle defined by top-left point x,y (included)
	 * and bottom-right point x+width-1,y+height-1 (included).
	 * <p>
	 * If either the given width or height is negative or zero, nothing is drawn.
	 *
	 * @param target
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the ellipse width.
	 * @param height
	 *            the ellipse height.
	 */
	public static void fillEllipse(byte[] target, int x, int y, int width, int height) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && width > 0 && height > 0) {
				// can disable clip when ellipse is fully in clip
				gc.configureClip(!gc.isRegionInClip(x, y, width, height));
				LLUIDisplay.Instance.getUIDrawer(gc).fillEllipse(gc, x, y, width, height);
			}
		}
	}

	/**
	 * MicroUI native "drawCircle". Draws a circle covering the square defined by top-left point x,y (included) and
	 * bottom-right point x+diameter-1,y+diameter-1 (included).
	 * <p>
	 * If the given diameter is negative or zero, nothing is drawn.
	 *
	 * @param target
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param diameter
	 *            the circle square size.
	 */
	public static void drawCircle(byte[] target, int x, int y, int diameter) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && diameter > 0) {
				// can disable clip when circle is fully in clip
				gc.configureClip(!gc.isRegionInClip(x, y, diameter, diameter));
				LLUIDisplay.Instance.getUIDrawer(gc).drawCircle(gc, x, y, diameter);
			}
		}
	}

	/**
	 * MicroUI native "fillCircle". Fills a circle covering the square defined by top-left point x,y (included) and
	 * bottom-right point x+diameter-1,y+diameter-1 (included).
	 * <p>
	 * If the given diameter is negative or zero, nothing is drawn.
	 *
	 * @param target
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param diameter
	 *            the circle square size.
	 */
	public static void fillCircle(byte[] target, int x, int y, int diameter) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			if (gc.requestDrawing() && diameter > 0) {
				// can disable clip when circle is fully in clip
				gc.configureClip(!gc.isRegionInClip(x, y, diameter, diameter));
				LLUIDisplay.Instance.getUIDrawer(gc).fillCircle(gc, x, y, diameter);
			}
		}
	}

	/**
	 * MicroUI native "drawImage". Draws a region of an image. The region of the image to draw is given relative to the
	 * image (origin at the upper-left corner) as a rectangle.
	 * <p>
	 * If the specified source region exceeds the image bounds, the copied region is limited to the image boundary. If
	 * the copied region goes out of the bounds of the graphics context area, pixels out of the range will not be drawn.
	 * <p>
	 * A global opacity value is given. When this value is 0xff (255, opaque), that means the image is drawn on the
	 * graphics context without managing an extra opacity. Only the image transparent pixels must have been merged with
	 * destination. All image opaque pixels override destination.
	 * <p>
	 * When this value is a value between 0 and 0xff, that means each pixel of the image must be merged with destination
	 * in addition with the image transparent pixels. An image opaque pixel becomes transparent (its opacity is the
	 * given alpha) and the opacity of an image transparent pixel becomes (alpha * alpha(pixel)) / 255.
	 *
	 *
	 * @param target
	 *            the MicroUI GraphicsContext target.
	 * @param source
	 *            the MicroUI Image to draw.
	 * @param regionX
	 *            the x coordinate of the upper-left corner of the region to copy.
	 * @param regionY
	 *            the y coordinate of the upper-left corner of the region to copy.
	 * @param width
	 *            the width of the region to copy.
	 * @param height
	 *            the height of the region to copy.
	 * @param x
	 *            the x coordinate of the top-left point in the destination.
	 * @param y
	 *            the y coordinate of the top-left point in the destination.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	public static void drawImage(byte[] target, byte[] source, int regionX, int regionY, int width, int height, int x,
			int y, int alpha) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			MicroUIImage image = graphicsEngine.mapMicroUIImage(source);
			if (gc.requestDrawing() && !image.isClosed() && alpha > 0) {
				image = graphicsEngine.getSource(image);
				// compute inside image bounds
				if (regionX < 0) {
					width += regionX; // decrease size
					x -= regionX; // increase origin
					regionX = 0;
				}
				if (regionX + width > image.getWidth()) {
					width = image.getWidth() - regionX; // decrease size
				}
				if (regionY < 0) {
					height += regionY; // decrease size
					y -= regionY; // increase origin
					regionY = 0;
				}
				if (regionY + height > image.getHeight()) {
					height = image.getHeight() - regionY; // decrease size
				}

				// compute inside graphics context bounds
				if (x < 0) {
					width += x; // decrease size
					regionX -= x; // increase origin
					x = 0;
				}
				if (x + width > gc.getWidth()) {
					width = gc.getWidth() - x; // decrease size
				}
				if (y < 0) {
					height += y; // decrease size
					regionY -= y; // increase origin
					y = 0;
				}
				if (y + height > gc.getHeight()) {
					height = gc.getHeight() - y; // decrease size
				}

				if (width > 0 && height > 0) {
					// can disable clip when region is fully in clip
					gc.configureClip(!gc.isRegionInClip(x, y, width, height));
					LLUIDisplay.Instance.getUIDrawer(gc).drawImage(gc, image, regionX, regionY, width, height, x, y,
							alpha);
				}
				// else: nothing to do
			}
			// else: refused drawing
		}
	}

	/**
	 * Draws a string.
	 *
	 * @param target
	 *            the targeted MicroUI GraphicsContext
	 * @param chars
	 *            a string of characters
	 * @param offset
	 *            the number of characters to skip, counting from the beginning
	 * @param length
	 *            the number of characters
	 * @param fontData
	 *            the MicroUI Font to use
	 * @param x
	 *            the left coordinate
	 * @param y
	 *            the top coordinate
	 */
	public static void drawString(byte[] target, char[] chars, int offset, int length, byte[] fontData, int x, int y) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			MicroUIFont font = graphicsEngine.mapMicroUIFont(fontData);
			char[] string = graphicsEngine.mapString(chars, offset, length);
			// TODO Handle the clip and check if the drawing is necessary
			if (gc.requestDrawing()) {
				LLUIDisplay.Instance.getUIDrawer(gc).drawString(gc, string, font, x, y);
			}
		}
	}

	/**
	 * Computes the rendered width of a string.
	 *
	 * @param chars
	 *            a string of characters
	 * @param offset
	 *            the number of characters to skip, counting from the beginning
	 * @param length
	 *            the number of characters
	 * @param fontData
	 *            the MicroUI Font to use
	 *
	 * @return the width in pixels
	 */
	public static int stringWidth(char[] chars, int offset, int length, byte[] fontData) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIFont font = graphicsEngine.mapMicroUIFont(fontData);
			char[] string = graphicsEngine.mapString(chars, offset, length);
			return LLUIDisplay.Instance.getUIFontDrawer(font).stringWidth(string, font);
		}
	}

	/**
	 * Draws a renderable string.
	 *
	 * @param target
	 *            the targeted MicroUI GraphicsContext
	 * @param chars
	 *            a string of characters
	 * @param offset
	 *            the number of characters to skip, counting from the beginning
	 * @param length
	 *            the number of characters
	 * @param fontData
	 *            the MicroUI Font to use
	 * @param charsWidth
	 *            the string width
	 * @param renderableStringSNIContext
	 *            the renderable string
	 * @param x
	 *            the left coordinate
	 * @param y
	 *            the top coordinate
	 */
	public static void drawRenderableString(byte[] target, char[] chars, // NOSONAR avoid to create object here
			int offset, int length, byte[] fontData, int charsWidth, byte[] renderableStringSNIContext, int x, int y) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIGraphicsContext gc = graphicsEngine.mapMicroUIGraphicsContext(target);
			MicroUIFont font = graphicsEngine.mapMicroUIFont(fontData);
			char[] string = graphicsEngine.mapString(chars, offset, length);
			MicroUIRenderableString renderableString = graphicsEngine
					.mapMicroUIRenderableString(renderableStringSNIContext);
			// TODO Handle the clip and check if the drawing is necessary
			if (gc.requestDrawing()) {
				LLUIDisplay.Instance.getUIDrawer(gc).drawRenderableString(gc, string, font, charsWidth,
						renderableString, x, y);
			}
		}
	}

	/**
	 * Computes the rendered width of a renderable string and fills its SNI context.
	 *
	 * @param chars
	 *            a string of characters
	 * @param offset
	 *            the number of characters to skip, counting from the beginning
	 * @param length
	 *            the number of characters
	 * @param fontData
	 *            the MicroUI Font to use
	 * @param renderableStringSNIContext
	 *            the renderable string
	 *
	 * @return the width in pixels
	 */
	public static int initializeRenderableStringSNIContext(char[] chars, int offset, int length, byte[] fontData,
			byte[] renderableStringSNIContext) {
		LLUIDisplay graphicsEngine = LLUIDisplay.Instance;
		synchronized (graphicsEngine) {
			MicroUIFont font = graphicsEngine.mapMicroUIFont(fontData);
			char[] string = graphicsEngine.mapString(chars, offset, length);
			MicroUIRenderableString renderableString = graphicsEngine
					.mapMicroUIRenderableString(renderableStringSNIContext);
			return LLUIDisplay.Instance.getUIFontDrawer(font).initializeRenderableStringSNIContext(string, font,
					renderableString);
		}
	}

}
