/*
 * Java
 *
 * Copyright 2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.bluetooth.util.server;

import ej.bluetooth.BluetoothDevice;
import ej.bluetooth.BluetoothPermissions;
import ej.bluetooth.BluetoothStatus;
import ej.bluetooth.util.BluetoothServicesUtil;

/**
 * A Client characteristic configuration descriptor.
 */
public class CCCDescriptor extends BasicServerDescriptor {

	/**
	 * Instantiates a {@link CCCDescriptor}.
	 */
	public CCCDescriptor() {
		super(BluetoothServicesUtil.CCC_UUID, BluetoothPermissions.RW);
	}

	@Override
	public void onReadRequest(BluetoothDevice device) {
		byte[] value = ServerStorage.get(device, this);
		if (value == null) {
			value = new byte[] { 0, 0 };
		}
		sendReadResponse(device, BluetoothStatus.OK, value);
	}

	@Override
	public void onWriteRequest(BluetoothDevice device, byte[] value) {
		if (value.length == 2) {
			ServerStorage.set(device, this, value);
			sendWriteResponse(device, BluetoothStatus.OK);
		} else {
			sendWriteResponse(device, BluetoothStatus.INVALID_VALUE_LENGTH);
		}
	}

	/**
	 * Checks whether a device should be notified.
	 *
	 * @param device
	 *            the device to check.
	 * @return <code>true</code> if the device should be notified.
	 */
	public boolean shouldNotify(BluetoothDevice device) {
		byte[] bs = ServerStorage.get(device, this);
		return bs != null && BluetoothServicesUtil.checkCccNotifications(bs);
	}

	/**
	 * Checks whether a device should be indicated.
	 *
	 * @param device
	 *            the device to check.
	 * @return <code>true</code> if the device should be indicated.
	 */
	public boolean shouldIndicate(BluetoothDevice device) {
		byte[] bs = ServerStorage.get(device, this);
		return bs != null && BluetoothServicesUtil.checkCccNIndication(bs);
	}
}
