/*
 * Java
 *
 * Copyright 2022 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import ej.fp.Image;
import ej.microui.display.LLUIPainter.MicroUIGraphicsContext;
import ej.microui.display.LLUIPainter.MicroUIImageFormat;

/**
 * Provides a service to create an object linked to a custom MicroUI BufferedImage (whose format is a value between
 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_CUSTOM_0} and
 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_CUSTOM_7}).
 * <p>
 * This kind of image is created when a MicroUI extension library offers the possibility to create a custom MicroUI
 * BufferedImage with a specific image format. This image may not hold a pixels buffer.
 * <p>
 * The Platform's Front Panel project (or a library) has to mention the implementation of this interface in a META-INF
 * file: <code>[src folder]/META-INF/services/ej.microui.display.BufferedImageProvider</code>:
 *
 * <pre>
 * ej.xxx.XXXProvider
 * </pre>
 *
 * where <code>XXXProvider</code> is an implementation of this interface.
 */
public interface BufferedImageProvider {

	/**
	 * Creates a new custom buffered image.
	 * <p>
	 * Note that a custom buffered image is an image which is not managed by the Graphics Engine' software algorithms.
	 * The Platform has the responsability to check if the drawing target is a custom image or not.
	 * <p>
	 * The Platform object can be retrieved at runtime by calling {@link Image#getRAWImage()}. All MicroUI (and derived)
	 * drawing methods must check if the target is a custom buffered image or not:
	 *
	 * <pre>
	 * public void fillRectangle(MicroUIGraphicsContext gc, int x1, int y1, int x2, int y2) {
	 * 	if (MicroUIImageFormat.MICROUI_IMAGE_FORMAT_CUSTOM_1 == gc.getFormat()) {
	 * 		Object customImage = gc.getImage().getRAWImage();
	 * 		// draw in a custom buffered image
	 * 		// ...
	 * 	} else {
	 * 		// performs the drawing as usual
	 * 		UIDrawingDefault.super.fillRectangle(gc, x1, y1, x2, y2);
	 * 	}
	 * }
	 * </pre>
	 *
	 * For the custom drawings, the {@link MicroUIGraphicsContext} (that targets a custom buffered image) can be used as
	 * a standard graphics context:
	 *
	 * <pre>
	 * public void drawSomething(byte[] target, int x1, int y1, int x2, int y2) {
	 * 	LLUIDisplay graphicalEngine = LLUIDisplay.Instance;
	 * 	synchronized (graphicalEngine) {
	 * 		MicroUIGraphicsContext gc = graphicalEngine.mapMicroUIGraphicsContext(target);
	 * 		if (gc.requestDrawing()) {
	 * 			if (MicroUIImageFormat.MICROUI_IMAGE_FORMAT_CUSTOM_1 == gc.getFormat()) {
	 * 				Object customImage = gc.getImage().getRAWImage();
	 * 				// draw in a custom buffered image
	 * 				// ...
	 * 			} else {
	 * 				// performs the drawing as usual
	 * 				// ...
	 * 			}
	 * 		}
	 * 	}
	 * }
	 * </pre>
	 *
	 * @param width
	 *            the custom buffered image width.
	 * @param height
	 *            the custom buffered image height.
	 * @param format
	 *            the custom buffered image format (between {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_CUSTOM_0} and
	 *            {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_CUSTOM_7}).
	 *
	 * @return an object that represents the custom buffered image
	 */
	Object newBufferedImage(int width, int height, MicroUIImageFormat format);

}
