/*
 * Java
 *
 * Copyright 2010-2015 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microui.display;

import ej.microui.util.EventHandler;

/**
 * <code>Displayable</code> is an abstract class which defines the very objects that can be shown on a
 * <code>Display</code>.<br>
 * A <code>Displayable</code> object is built for a specific <code>Display</code> which can not be changed afterwards. A
 * <code>Displayable</code> may be shown or hidden, but at most one <code>Displayable</code> is shown per
 * <code>Display</code>.<br>
 * <br>
 * Subclasses should define the <code>Displayable</code> contents and their possible interactions with the user.<br>
 * <br>
 * By default, a new <code>Displayable</code> object is not visible on its display.
 * 
 * @see Display
 */
public abstract class Displayable {

	/**
	 * The newly created displayable is built for the given display and is hidden.
	 * 
	 * @param display
	 *            the display for which the displayable is created
	 * @throws NullPointerException
	 *             if the given display is <code>null</code>
	 */
	public Displayable(Display display) {
		throw new RuntimeException();
	}

	/**
	 * Gets the displayable's display.
	 * <p>
	 * The returned display can't be <code>null</code>.
	 * 
	 * @return the displayable's display.
	 */
	public Display getDisplay() {
		throw new RuntimeException();
	}

	/**
	 * Checks whether the displayable is visible on its display.
	 * 
	 * @return <code>true</code> if the displayable is currently visible, <code>false</code> otherwise
	 */
	public boolean isShown() {
		throw new RuntimeException();
	}

	/**
	 * Sets the displayable as visible on its display.
	 * 
	 * @throws SecurityException
	 *             if a security manager exists and does not allow the caller to get the display.
	 */
	public void show() throws SecurityException {
		throw new RuntimeException();
	}

	/**
	 * Sets the displayable as hidden on its display. If the displayable is not visible, this method has no effect.
	 */
	public void hide() {
		throw new RuntimeException();
	}

	/**
	 * This method is called by system as soon as the displayable becomes visible. Application should override this
	 * method to control its own displayables.
	 */
	protected void showNotify() {
		throw new RuntimeException();
	}

	/**
	 * This method is called by system as soon as the displayable becomes hidden. Application should override this
	 * method to control its own displayables.
	 */
	protected void hideNotify() {
		throw new RuntimeException();
	}

	/**
	 * Requests a repaint for the entire displayable. Calling this method may result in subsequent call(s) to
	 * {@link #paint(GraphicsContext)} on the displayable.
	 * <p>
	 * If the displayable is not visible, this call has no effect.
	 * <p>
	 * The call(s) to {@link #paint(GraphicsContext)} occurs asynchronously to this call. That is, this method will not
	 * block waiting for {@link #paint(GraphicsContext)} to finish.
	 * <p>
	 * To synchronize with the {@link #paint(GraphicsContext)} routine, applications can use either
	 * {@link Display#callSerially(Runnable)} or {@link Display#waitForEvent()}, or they can code explicit
	 * synchronization into their {@link #paint(GraphicsContext)} routine.
	 */
	public void repaint() {
		throw new RuntimeException();
	}

	/**
	 * Draws the displayable. This method must be implemented by subclasses to render graphics on a display.
	 * 
	 * @param g
	 *            the {@link GraphicsContext} of the displayable's {@link Display}.
	 */
	public abstract void paint(GraphicsContext g);

	/**
	 * Gets the displayable controller or <code>null</code> if none.
	 * 
	 * @return the displayable controller or <code>null</code>.
	 * @since 2.0
	 */
	public abstract EventHandler getController();

}
