/*
 * Java
 *
 * Copyright 2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg;

import ej.microui.display.MicroUIGraphicsContext;
import ej.trace.Tracer;

/**
 * Utility class to log MicroVG drawing events.
 * <p>
 * Log parameters are associated with the MicroVG SystemView description file <code>SYSVIEW_MicroVG.txt</code>. If an
 * event is malformed, the log will not be correctly decoded.
 */
public class Log {

	/**
	 * Event identifier for "draw a path with a color".
	 */
	public static final int DRAW_PATH_COLOR = 10;

	/**
	 * Event identifier for "draw a path with a gradient".
	 */
	public static final int DRAW_PATH_GRADIENT = 11;

	/**
	 * Event identifier for "draw a string with a color".
	 */
	public static final int DRAW_STRING_COLOR = 12;

	/**
	 * Event identifier for "draw a string with a gradient".
	 */
	public static final int DRAW_STRING_GRADIENT = 13;

	/**
	 * Event identifier for "draw a string on a circle with a color".
	 */
	public static final int DRAW_STRING_ON_CIRCLE_COLOR = 14;

	/**
	 * Event identifier for "draw a string on a circle with a gradient".
	 */
	public static final int DRAW_STRING_ON_CIRCLE_GRADIENT = 15;

	/**
	 * Event identifier for "draw an image".
	 */
	public static final int DRAW_IMAGE = 16;

	private static final Tracer MicroVGTracer = new Tracer("MicroVG", 40); //$NON-NLS-1$

	private Log() {
		// cannot instantiate
	}

	/**
	 * Records a MicroVG event. The given event identifier must be valid for this logger (i.e. between
	 * {@link #DRAW_PATH_COLOR} and {@link #DRAW_IMAGE}).
	 *
	 * @param eventId
	 *            the event identifier
	 * @param values
	 *            the event parameters
	 */
	public static void recordEvent(int eventId, int... values) {
		MicroVGTracer.recordEvent(eventId, values);
	}

	/**
	 * Records the end of the execution of a MicroVG event. Call this method to trace the duration of an event
	 * previously record with one of the {@link #recordEventEnd(int, int...)} method. This method should not be called
	 * for instantaneous events.
	 *
	 * @param eventId
	 *            the "end" event identifier
	 * @param values
	 *            the "end" event parameters
	 */
	public static void recordEventEnd(int eventId, int... values) {
		MicroVGTracer.recordEventEnd(eventId, values);
	}

	/**
	 * Maps a graphics context to an event parameter.
	 *
	 * @param gc
	 *            the graphics context to log
	 * @return the unique graphics context identifier
	 */
	public static int mapGraphicsContext(MicroUIGraphicsContext gc) {
		return ej.microui.Log.mapGraphicsContext(gc);
	}

	/**
	 * Maps an image to an event parameter.
	 *
	 * @param img
	 *            the image to log
	 * @return the unique image identifier
	 */
	public static int mapImage(MicroVGImage img) {
		return img.getIdentifier();
	}
}
