/*
 * Protocol Buffers - Google's data interchange format
 * Copyright 2008 Google Inc.  All rights reserved.
 * https: *developers.google.com/protocol-buffers/
 * Copyright 2020 MicroEJ Corp. This file has been modified by MicroEJ Corp.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *     * Neither the name of Google Inc. nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.google.protobuf;

import java.util.Arrays;
import java.util.Collection;
import java.util.RandomAccess;

import com.google.protobuf.Internal.FloatList;

/**
 * An implementation of {@link FloatList} on top of a primitive array.
 *
 * @author dweis@google.com (Daniel Weis)
 */
final class FloatArrayList extends AbstractProtobufList<Float> implements FloatList, RandomAccess {

	private static final FloatArrayList EMPTY_LIST = new FloatArrayList();
	static {
		EMPTY_LIST.makeImmutable();
	}

	public static FloatArrayList emptyList() {
		return EMPTY_LIST;
	}

	/**
	 * The backing store for the list.
	 */
	private float[] array;

	/**
	 * The size of the list distinct from the length of the array. That is, it is the number of elements set in the
	 * list.
	 */
	private int size;

	/**
	 * Constructs a new mutable {@code FloatArrayList} with default capacity.
	 */
	FloatArrayList() {
		this(new float[DEFAULT_CAPACITY], 0);
	}

	/**
	 * Constructs a new mutable {@code FloatArrayList} containing the same elements as {@code other}.
	 */
	private FloatArrayList(float[] other, int size) {
		this.array = other;
		this.size = size;
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (!(o instanceof FloatArrayList)) {
			return super.equals(o);
		}
		FloatArrayList other = (FloatArrayList) o;
		if (this.size != other.size) {
			return false;
		}

		final float[] arr = other.array;
		for (int i = 0; i < this.size; i++) {
			if (this.array[i] != arr[i]) {
				return false;
			}
		}

		return true;
	}

	@Override
	public int hashCode() {
		int result = 1;
		for (int i = 0; i < this.size; i++) {
			result = (31 * result) + Float.floatToIntBits(this.array[i]);
		}
		return result;
	}

	@Override
	public FloatList mutableCopyWithCapacity(int capacity) {
		if (capacity < this.size) {
			throw new IllegalArgumentException();
		}
		return new FloatArrayList(Arrays.copyOf(this.array, capacity), this.size);
	}

	@Override
	public Float get(int index) {
		return Float.valueOf(getFloat(index));
	}

	@Override
	public float getFloat(int index) {
		ensureIndexInRange(index);
		return this.array[index];
	}

	@Override
	public int size() {
		return this.size;
	}

	@Override
	public Float set(int index, Float element) {
		return Float.valueOf(setFloat(index, element.floatValue()));
	}

	@Override
	public float setFloat(int index, float element) {
		ensureIsMutable();
		ensureIndexInRange(index);
		float previousValue = this.array[index];
		this.array[index] = element;
		return previousValue;
	}

	@Override
	public void add(int index, Float element) {
		addFloat(index, element.floatValue());
	}

	/**
	 * Like {@link #add(Float)} but more efficient in that it doesn't box the element.
	 */
	@Override
	public void addFloat(float element) {
		addFloat(this.size, element);
	}

	/**
	 * Like {@link #add(int, Float)} but more efficient in that it doesn't box the element.
	 */
	private void addFloat(int index, float element) {
		ensureIsMutable();
		if (index < 0 || index > this.size) {
			throw new IndexOutOfBoundsException(makeOutOfBoundsExceptionMessage(index));
		}

		if (this.size < this.array.length) {
			// Shift everything over to make room
			System.arraycopy(this.array, index, this.array, index + 1, this.size - index);
		} else {
			// Resize to 1.5x the size
			int length = ((this.size * 3) / 2) + 1;
			float[] newArray = new float[length];

			// Copy the first part directly
			System.arraycopy(this.array, 0, newArray, 0, index);

			// Copy the rest shifted over by one to make room
			System.arraycopy(this.array, index, newArray, index + 1, this.size - index);
			this.array = newArray;
		}

		this.array[index] = element;
		this.size++;
		this.modCount++;
	}

	@Override
	public boolean addAll(Collection<? extends Float> collection) {
		ensureIsMutable();

		if (collection == null) {
			throw new NullPointerException();
		}

		// We specialize when adding another FloatArrayList to avoid boxing elements.
		if (!(collection instanceof FloatArrayList)) {
			return super.addAll(collection);
		}

		FloatArrayList list = (FloatArrayList) collection;
		if (list.size == 0) {
			return false;
		}

		int overflow = Integer.MAX_VALUE - this.size;
		if (overflow < list.size) {
			// We can't actually represent a list this large.
			throw new OutOfMemoryError();
		}

		int newSize = this.size + list.size;
		if (newSize > this.array.length) {
			this.array = Arrays.copyOf(this.array, newSize);
		}

		System.arraycopy(list.array, 0, this.array, this.size, list.size);
		this.size = newSize;
		this.modCount++;
		return true;
	}

	@Override
	public boolean remove(Object o) {
		ensureIsMutable();
		for (int i = 0; i < this.size; i++) {
			if (o.equals(Float.valueOf(this.array[i]))) {
				System.arraycopy(this.array, i + 1, this.array, i, this.size - i);
				this.size--;
				this.modCount++;
				return true;
			}
		}
		return false;
	}

	@Override
	public Float remove(int index) {
		ensureIsMutable();
		ensureIndexInRange(index);
		float value = this.array[index];
		if (index < this.size - 1) {
			System.arraycopy(this.array, index + 1, this.array, index, this.size - index - 1);
		}
		this.size--;
		this.modCount++;
		return Float.valueOf(value);
	}

	/**
	 * Ensures that the provided {@code index} is within the range of {@code [0, size]}. Throws an
	 * {@link IndexOutOfBoundsException} if it is not.
	 *
	 * @param index
	 *            the index to verify is in range
	 */
	private void ensureIndexInRange(int index) {
		if (index < 0 || index >= this.size) {
			throw new IndexOutOfBoundsException(makeOutOfBoundsExceptionMessage(index));
		}
	}

	private String makeOutOfBoundsExceptionMessage(int index) {
		return "Index:" + index + ", Size:" + this.size;
	}
}
