/*
 * Java
 *
 * Copyright 2021-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microvg.image;

/**
 * This class offers matrix services.
 *
 * The implementation does not create any object.
 */
public class MatrixHelper {

	private MatrixHelper() {
		// Prevent instantiation.
	}

	/**
	 * Creates a matrix.
	 *
	 * @return the new matrix
	 */
	public static float[] create() {
		return new float[9];
	}

	/**
	 * Creates an identity matrix.
	 *
	 * @return the new matrix
	 */
	public static float[] createIdentity() {
		float[] matrix = create();
		identity(matrix);
		return matrix;
	}

	/**
	 * Creates an identity matrix translated to (x,y) .
	 *
	 * @param x
	 *            the translation on x
	 * @param y
	 *            the translation on y
	 * @return the new matrix
	 */
	public static float[] create(float x, float y) {
		float[] matrix = create();
		reset(matrix, x, y);
		return matrix;
	}

	/**
	 * Updates the matrix as identity matrix.
	 *
	 * @param matrix
	 *            the matrix to initialize
	 */
	public static void identity(float[] matrix) {
		reset(matrix, 0, 0);
	}

	/**
	 * Resets the matrix as identity matrix and translates it.
	 *
	 * @param matrix
	 *            the matrix to initialize
	 * @param x
	 *            the translation on x
	 * @param y
	 *            the translation on y
	 */
	public static void reset(float[] matrix, float x, float y) {
		matrix[0] = 1.0f;
		matrix[1] = 0.0f;
		matrix[2] = x;
		matrix[3] = 0.0f;
		matrix[4] = 1.0f;
		matrix[5] = y;
		matrix[6] = 0.0f;
		matrix[7] = 0.0f;
		matrix[8] = 1.0f;
	}

	/**
	 * Deep copies the given matrix into destination.
	 *
	 * @param dest
	 *            the matrix to copy to
	 * @param src
	 *            the matrix to copy from
	 */
	public static void copy(float[] dest, float[] src) {
		System.arraycopy(src, 0, dest, 0, dest.length);
	}

	/**
	 * Preconcats the matrix with the specified translation. M' = M * T(dx, dy)
	 *
	 * @param matrix
	 *            the matrix to translate
	 * @param x
	 *            the translation on x
	 * @param y
	 *            the translation on y
	 */
	public static void translate(float[] matrix, float x, float y) {
		matrix[2] = matrix[0] * x + matrix[1] * y + matrix[2];
		matrix[5] = matrix[3] * x + matrix[4] * y + matrix[5];
		matrix[8] = matrix[6] * x + matrix[7] * y + matrix[8];
	}

	/**
	 * Preconcats the matrix with the specified scale. M' = M * S(sx, sy)
	 *
	 * @param matrix
	 *            the matrix to scale
	 * @param scaleX
	 *            the scale on x
	 * @param scaleY
	 *            the scale on y
	 */
	public static void scale(float[] matrix, float scaleX, float scaleY) {
		matrix[0] *= scaleX;
		matrix[1] *= scaleY;
		matrix[3] *= scaleX;
		matrix[4] *= scaleY;
		matrix[6] *= scaleX;
		matrix[7] *= scaleY;
	}

	/**
	 * Preconcats the matrix with the specified rotation. M' = M * R(degrees)
	 *
	 * @param matrix
	 *            the matrix to rotate
	 * @param degrees
	 *            the degrees to rotate
	 */
	public static void rotate(float[] matrix, float degrees) {
		final double angleRadians = Math.toRadians(degrees);

		float cosAngle = (float) Math.cos(angleRadians);
		float sinAngle = (float) Math.sin(angleRadians);

		float tmp;

		tmp = cosAngle * matrix[0] + sinAngle * matrix[1];
		matrix[1] = cosAngle * matrix[1] - sinAngle * matrix[0];
		matrix[0] = tmp;

		tmp = cosAngle * matrix[3] + sinAngle * matrix[4];
		matrix[4] = cosAngle * matrix[4] - sinAngle * matrix[3];
		matrix[3] = tmp;

		tmp = cosAngle * matrix[6] + sinAngle * matrix[7];
		matrix[7] = cosAngle * matrix[7] - sinAngle * matrix[6];
		matrix[6] = tmp;
	}

	/**
	 * Preconcats the matrix with the specified matrix. M' = M * other
	 *
	 * @param matrix
	 *            the matrix to concat
	 * @param other
	 *            the matrix to concat with
	 */
	public static void concatenate(float[] matrix, float[] other) {
		float[] temp = create();

		temp[0] = (matrix[0] * other[0]) + (matrix[1] * other[3]) + (matrix[2] * other[6]);
		temp[1] = (matrix[0] * other[1]) + (matrix[1] * other[4]) + (matrix[2] * other[7]);
		temp[2] = (matrix[0] * other[2]) + (matrix[1] * other[5]) + (matrix[2] * other[8]);

		temp[3] = (matrix[3] * other[0]) + (matrix[4] * other[3]) + (matrix[5] * other[6]);
		temp[4] = (matrix[3] * other[1]) + (matrix[4] * other[4]) + (matrix[5] * other[7]);
		temp[5] = (matrix[3] * other[2]) + (matrix[4] * other[5]) + (matrix[5] * other[8]);

		temp[6] = (matrix[6] * other[0]) + (matrix[7] * other[3]) + (matrix[8] * other[6]);
		temp[7] = (matrix[6] * other[1]) + (matrix[7] * other[4]) + (matrix[8] * other[7]);
		temp[8] = (matrix[6] * other[2]) + (matrix[7] * other[5]) + (matrix[8] * other[8]);

		/* Copy temporary matrix into result. */
		copy(matrix, temp);
	}
}
