/*
 * Java
 *
 * Copyright 2021-2025 MicroEJ Corp.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 *
 * This File is a derivative work. Subject to §4 of the applicable Apache License, MicroEJ provides the above different license terms and conditions for use.
 *
 * // Copyright (C) 2006 The Android Open Source Project
 * //
 * // Licensed under the Apache License, Version 2.0 (the "License");
 * // you may not use this file except in compliance with the License.
 * // You may obtain a copy of the License at
 * //
 * //      http://www.apache.org/licenses/LICENSE-2.0
 * //
 * // Unless required by applicable law or agreed to in writing, software
 * // distributed under the License is distributed on an "AS IS" BASIS,
 * // WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * // See the License for the specific language governing permissions and
 * // limitations under the License.
 */
package ej.microvg;

/**
 * Enum for the ways a path is blend.
 */
public enum BlendMode {
	/**
	 * The source pixels replace the destination pixels.
	 * <p>
	 * \(\alpha_{out} = \alpha_{src}\)
	 * <p>
	 * \(C_{out} = C_{src}\)
	 */
	SRC,

	/**
	 * The source pixels are drawn over the destination pixels.
	 * <p>
	 * \(\alpha_{out} = \alpha_{src} + (1 - \alpha_{src}) * \alpha_{dst}\)
	 * <p>
	 * \(C_{out} = C_{src} + (1 - \alpha_{src}) * C_{dst}\)
	 */
	SRC_OVER,

	/**
	 * The source pixels are drawn behind the destination pixels.
	 * <p>
	 * \(\alpha_{out} = \alpha_{dst} + (1 - \alpha_{dst}) * \alpha_{src}\)
	 * <p>
	 * \(C_{out} = C_{dst} + (1 - \alpha_{dst}) * C_{src}\)
	 */
	DST_OVER,

	/**
	 * Keeps the source pixels that cover the destination pixels, discards the remaining source and destination pixels.
	 * <p>
	 * \(\alpha_{out} = \alpha_{src} * \alpha_{dst}\)
	 * <p>
	 * \(C_{out} = C_{src} * \alpha_{dst}\)
	 */
	SRC_IN,

	/**
	 * Keeps the destination pixels that cover source pixels, discards the remaining source and destination pixels.
	 * <p>
	 * \(\alpha_{out} = \alpha_{src} * \alpha_{dst}\)
	 * <p>
	 * \(C_{out} = C_{dst} * \alpha_{src}\)
	 */
	DST_IN,

	/**
	 * Keeps the destination pixels that are not covered by source pixels. Discards destination pixels that are covered
	 * by source pixels. Discards all source pixels.
	 * <p>
	 * \(\alpha_{out} = (1 - \alpha_{src}) * \alpha_{dst}\)
	 * </p>
	 * <p>
	 * \(C_{out} = (1 - \alpha_{src}) * C_{dst}\)
	 * </p>
	 */
	DST_OUT,

	/**
	 * Adds the source pixels to the destination pixels and saturates the result.
	 * <p>
	 * \(\alpha_{out} = max(0, min(\alpha_{src} + \alpha_{dst}, 1))\)
	 * </p>
	 * <p>
	 * \(C_{out} = max(0, min(C_{src} + C_{dst}, 1))\)
	 * </p>
	 */
	PLUS,

	/**
	 * Adds the source and destination pixels, then subtracts the source pixels multiplied by the destination.
	 * <p>
	 * \(\alpha_{out} = \alpha_{src} + \alpha_{dst} - \alpha_{src} * \alpha_{dst}\)
	 * <p>
	 * \(C_{out} = C_{src} + C_{dst} - C_{src} * C_{dst}\)
	 */
	SCREEN,

	/**
	 * Multiplies the source and destination pixels.
	 * <p>
	 * \(\alpha_{out} = \alpha_{src} + \alpha_{dst} - \alpha_{src} * \alpha_{dst}\)
	 * <p>
	 * \(C_{out} = C_{src} * (1 - \alpha_{dst}) + C_{dst} * (1 - \alpha_{src}) + (C_{src} * C_{dst})\)
	 */
	MULTIPLY,
}
