/*
 * Java
 *
 * Copyright 2009-2019 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.fp.util;

/**
 * This thread is used to send an event on demand.
 * <p>
 * The thread has be waked up using {@link #wakeup()} method. After a time defined by the {@link ThirdEventWidget}, the
 * thread asks to the {@link ThirdEventWidget} to send the event.
 * <p>
 * At any moment the thread can stop event timeout calling the method {@link #goToSleep()}.
 */
public class ThirdEventThread extends Thread {

	/**
	 * The {@link ThirdEventThread} requires an implementation of {@link ThirdEventWidget} to send to event after a
	 * sleep period.
	 */
	public static interface ThirdEventWidget {

		/**
		 * Gets the sleep period the thread has to wait before asking to send third event.
		 *
		 * @return a time in milliseconds.
		 */
		int getSleepPeriod();

		/**
		 * Asks to send the third event.
		 */
		void sendThirdEvent();
	}

	private final ThirdEventWidget widget;
	private final Object monitor;
	private final boolean infinite;
	private boolean sendThirdEvent;
	private boolean isRunning;

	/**
	 * Creates a ThirdEventThread for the ThirdEventWidget.
	 *
	 * @param widget
	 *            the widget which sends the third event.
	 * @param infinite
	 *            true to send the third event indifinitively until {@link #goToSleep()} is called.
	 */
	public ThirdEventThread(ThirdEventWidget widget, boolean infinite) {
		this.widget = widget;
		this.infinite = infinite;
		this.monitor = new Object();
		this.isRunning = true;
	}

	/**
	 * Disposes this thread stopping it.
	 */
	public void dispose() {
		this.isRunning = false;
		goToSleep();
	}

	/**
	 * Wakes up the thread: it will waits during waiting period before send the third event.
	 */
	public void wakeup() {
		synchronized (this.monitor) {
			this.sendThirdEvent = true;
			this.monitor.notifyAll();
		}
	}

	/**
	 * Sleeps the thread.
	 */
	public void goToSleep() {
		synchronized (this.monitor) {
			this.sendThirdEvent = false;
			this.monitor.notifyAll();
		}
	}

	/**
	 * Returns true when a call to {@link #wakeup()} has been performed whereas a call to {@link #goToSleep()} not.
	 *
	 * @return true when thread is ready to send a third event.
	 */
	public boolean isSendingThirdEvent() {
		return this.sendThirdEvent;
	}

	@Override
	public void run() {

		ThirdEventWidget widget = this.widget;

		while (this.isRunning) {

			synchronized (this.monitor) {

				if (!this.sendThirdEvent) {
					// sleep the thread
					try {
						this.monitor.wait();
					} catch (InterruptedException e) {
						// end of thread
						Thread.currentThread().interrupt();
						return;
					}
				}

				if (this.sendThirdEvent) {
					// need to send a third event AFTER the sleep period
					try {
						this.monitor.wait(widget.getSleepPeriod());
					} catch (InterruptedException e) {
						// end of thread
						Thread.currentThread().interrupt();
						return;
					}
				}
			}

			if (this.sendThirdEvent) {
				widget.sendThirdEvent();

				// update sendThirdEvent
				this.sendThirdEvent = this.infinite;
			}
		}

	}
}
