/*
 * Java
 *
 * Copyright 2023-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microvg;

import java.awt.geom.GeneralPath;

import ej.microui.display.LLUIPainter;
import ej.microui.display.MicroUIGraphicsContext;
import ej.microui.display.MicroUIImageFormat;
import ej.microvg.paint.VGPaint;

/**
 * This interface provides all drawing methods useful to implement MicroVG drawing native methods.
 */
public interface VGDrawing {

	/**
	 * Gets the supported buffered image format.
	 *
	 * @return the supported buffered image format
	 */
	MicroUIImageFormat handledFormat();

	/**
	 * Draws a general path filled with the given color representation.
	 *
	 * @param gc
	 *            the destination
	 * @param path
	 *            the path to draw
	 * @param x
	 *            the destination X coordinate
	 * @param y
	 *            the destination Y coordinate
	 * @param matrix
	 *            the deformation to apply
	 * @param fillRule
	 *            the fill type
	 * @param blend
	 *            the blend mode
	 * @param paint
	 *            the color representation to use
	 */
	default void drawPath(MicroUIGraphicsContext gc, GeneralPath path, float x, float y, float[] matrix, int fillRule,
			int blend, VGPaint paint) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws an image with transformation and opacity. Optionally apply an animation and / or a color filtering.
	 *
	 * @param gc
	 *            the graphics context to draw on
	 * @param image
	 *            the image to draw
	 * @param matrix
	 *            the transformation matrix to apply
	 * @param alpha
	 *            the global opacity rendering value
	 * @param elapsedTime
	 *            the elapsed time since the beginning of the animation, in milliseconds
	 * @param colorMatrix
	 *            the color matrix used to transform colors
	 * @return a VectorGraphicsException error code
	 */
	default int drawImage(MicroUIGraphicsContext gc, MicroVGImage image, float[] matrix, int alpha, long elapsedTime,
			float[] colorMatrix) {
		// default implementation : let the image draw itself on the destination
		return image.draw(gc, matrix, alpha, elapsedTime, colorMatrix);
	}

	/**
	 * Draws a string with the given painter. Default implementation redirects this drawing to the
	 * {@link LLVGEngine#drawString(MicroUIGraphicsContext, char[], int, float, float, float, float[], int, VGPaint, float, float, int)}
	 * method.
	 *
	 * @param gc
	 *            the destination
	 * @param string
	 *            the array of characters to draw
	 * @param face
	 *            the font face to use
	 * @param size
	 *            the font size, in pixels
	 * @param x
	 *            the destination X coordinate
	 * @param y
	 *            the destination Y coordinate
	 * @param matrix
	 *            the deformation to apply
	 * @param blendMode
	 *            the blend mode
	 * @param p
	 *            the color representation to use
	 * @param letterSpacing
	 *            the extra letter spacing to use
	 * @param radius
	 *            the radius of the circle
	 * @param direction
	 *            the direction of the text along the circle
	 * @return {@link LLVGPainter#RETURN_SUCCESS} on success or an error code
	 */
	default int drawString(MicroUIGraphicsContext gc, char[] string, int face, float size, float x, float y,
			float[] matrix, int blendMode, VGPaint p, float letterSpacing, float radius, int direction) {
		return LLVGEngine.Instance.drawString(gc, string, face, size, x, y, matrix, blendMode, p, letterSpacing, radius,
				direction);
	}
}
