/*
 * Copyright 2020-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.widget.util.debug;

import ej.mwt.Container;
import ej.mwt.Desktop;
import ej.mwt.Widget;

/**
 * Provides helpers to analyze a hierarchy of widgets.
 */
public class HierarchyInspector {

	// Prevents initialization.
	private HierarchyInspector() {
	}

	/**
	 * Counts the number of containers in the child hierarchy of a widget.
	 * <p>
	 * The method is recursive: if the given widget is a container, it browses its children.
	 *
	 * @param widget
	 *            the root widget.
	 * @return the number of containers.
	 */
	public static int countNumberOfContainers(Widget widget) {
		int number = 0;
		if (widget instanceof Container) {
			number++;
			Container container = (Container) widget;
			int numChildren = container.getChildrenCount();
			for (int i = 0; i < numChildren; i++) {
				Widget child = container.getChild(i);
				number += countNumberOfContainers(child);
			}
		}
		return number;
	}

	/**
	 * Counts the number of widgets in the child hierarchy of a widget.
	 * <p>
	 * The method is recursive: if the given widget is a container, it browses its children.
	 *
	 * @param widget
	 *            the root widget.
	 * @return the number of containers.
	 */
	public static int countNumberOfWidgets(Widget widget) {
		int number = 0;
		if (widget instanceof Container) {
			Container container = (Container) widget;
			int numChildren = container.getChildrenCount();
			for (int i = 0; i < numChildren; i++) {
				Widget child = container.getChild(i);
				number += countNumberOfWidgets(child);
			}
		} else {
			number++;
		}
		return number;
	}

	/**
	 * Counts the maximum depth of the child hierarchy of a widget.
	 * <p>
	 * The method is recursive: if the given widget is a container, it browses its children.
	 *
	 * @param widget
	 *            the root widget.
	 * @return the maximum depth.
	 */
	public static int countMaxDepth(Widget widget) {
		return countMaxDepth(widget, 0);
	}

	private static int countMaxDepth(Widget widget, int currentDepth) {
		int maxDepth = currentDepth;
		if (widget instanceof Container) {
			Container container = (Container) widget;
			int numChildren = container.getChildrenCount();
			for (int i = 0; i < numChildren; i++) {
				Widget child = container.getChild(i);
				int childDepth = countMaxDepth(child, currentDepth + 1);
				maxDepth = Math.max(maxDepth, childDepth);
			}
		}
		return maxDepth;
	}

	/**
	 * Prints the hierarchy of the given widget.
	 * <p>
	 * Prints the widget and its children recursively in a tree format.
	 *
	 * @param widget
	 *            the widget.
	 * @return the hierarchy.
	 */
	public static String hierarchyToString(Widget widget) {
		StringBuilder builder = new StringBuilder();

		appendHierarchyRecursive(builder, widget, 0);

		return builder.toString();
	}

	private static void appendHierarchyRecursive(StringBuilder builder, Widget widget, int depth) {
		appendDepth(builder, depth);
		appendElement(builder, widget);
		builder.append('\n');
		if (widget instanceof Container) {
			appendChildrenHierarchy(builder, (Container) widget, depth);
		}
	}

	private static void appendChildrenHierarchy(StringBuilder builder, Container container, int depth) {
		int childrenCount = container.getChildrenCount();
		for (int i = 0; i < childrenCount; i++) {
			Widget child = container.getChild(i);
			appendHierarchyRecursive(builder, child, depth + 1);
		}
	}

	static StringBuilder appendDepth(StringBuilder builder, int depth) {
		while (depth > 1) {
			builder.append('|').append(' ').append(' ');
			depth--;
		}
		if (depth > 0) {
			builder.append('+').append('-').append('-');
		}
		return builder;
	}

	/**
	 * Prints the path to the given widget with <code>&gt;</code> separator.
	 *
	 * @param widget
	 *            the widget to inspect.
	 * @return the path.
	 * @see HierarchyInspector#pathToWidgetToString(Widget, char)
	 */
	public static String pathToWidgetToString(Widget widget) {
		return pathToWidgetToString(widget, '>');
	}

	/**
	 * Prints the path to the given widget.
	 * <p>
	 * Prints the containers from the root (including the desktop) to the widget, separated with the given separator.
	 * Example: <code>Desktop &gt; Scroll &gt; ScrollableList &gt; Label</code>
	 *
	 * @param widget
	 *            the widget to inspect.
	 * @param separator
	 *            the separator between the items.
	 * @return the path.
	 */
	public static String pathToWidgetToString(Widget widget, char separator) {
		StringBuilder builder = new StringBuilder();

		if (widget.isAttached()) {
			Desktop desktop = widget.getDesktop();
			appendElement(builder, desktop);
			appendSeparator(builder, separator);
		}

		pathToWidgetRecursive(builder, widget, separator);

		return builder.toString();
	}

	/**
	 * Prints the root parent first then all the children.
	 */
	private static void pathToWidgetRecursive(StringBuilder builder, Widget widget, char separator) {
		Container parent = widget.getParent();
		if (parent != null) {
			pathToWidgetRecursive(builder, parent, separator);
			appendSeparator(builder, separator);
		}
		appendElement(builder, widget);
	}

	static StringBuilder appendElement(StringBuilder builder, Object object) {
		String simpleName = object.getClass().getSimpleName();
		int length = simpleName.length();
		if (simpleName.charAt(length - 1) == '@' && simpleName.charAt(0) != '@') {
			simpleName = simpleName.substring(0, length - 1);
		}
		builder.append(simpleName);
		return builder;
	}

	private static void appendSeparator(StringBuilder builder, char separator) {
		builder.append(' ').append(separator).append(' ');
	}

}
