/*
 * Copyright 2017-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.widget.util.color;

import ej.bon.XMath;
import ej.microui.display.Display;

/**
 * Gradient utilities.
 *
 * @since 2.3.0
 */
public class GradientHelper {

	// Prevents initialization.
	private GradientHelper() {
	}

	/**
	 * Computes the color resulting by blending two colors with a ratio.
	 *
	 * @param color1
	 *            the first color.
	 * @param color2
	 *            the second color.
	 * @param ratio
	 *            the ratio of the second color which means the ratio of the first color is <code>1 - ratio</code>. This
	 *            value is clamped between 0 and 1.
	 * @return the blended color.
	 */
	public static int blendColors(int color1, int color2, float ratio) {
		ratio = XMath.limit(ratio, 0.0f, 1.0f);

		int red1 = ColorHelper.getRed(color1);
		int green1 = ColorHelper.getGreen(color1);
		int blue1 = ColorHelper.getBlue(color1);
		int red2 = ColorHelper.getRed(color2);
		int green2 = ColorHelper.getGreen(color2);
		int blue2 = ColorHelper.getBlue(color2);

		float invRatio = 1.0f - ratio;
		int red = (int) (red1 * invRatio + red2 * ratio);
		int green = (int) (green1 * invRatio + green2 * ratio);
		int blue = (int) (blue1 * invRatio + blue2 * ratio);
		return ColorHelper.getColor(red, green, blue);
	}

	/**
	 * Creates a gradient between two colors on a display.
	 * <p>
	 * The result is an array of colors where:
	 * <ul>
	 * <li>the first one is the given start color,</li>
	 * <li>the last one is the given end color,</li>
	 * <li>all colors are different.</li>
	 * </ul>
	 * Beware that if the given colors are identical, the result contains only one element.
	 *
	 * @param display
	 *            the display where the gradient will be used.
	 * @param startColor
	 *            the start color of the gradient.
	 * @param endColor
	 *            the end color of the gradient.
	 * @return the gradient.
	 * @see Display#getDisplayColor(int)
	 */
	public static int[] createGradient(Display display, int startColor, int endColor) {
		// Gets color components.
		float currentRed = ColorHelper.getRed(startColor);
		float currentGreen = ColorHelper.getGreen(startColor);
		float currentBlue = ColorHelper.getBlue(startColor);
		int endRed = ColorHelper.getRed(endColor);
		int endGreen = ColorHelper.getGreen(endColor);
		int endBlue = ColorHelper.getBlue(endColor);

		// Computes the max number of steps for the array and for the progress factor.
		int stepsRed = (int) (endRed - currentRed);
		int stepsGreen = (int) (endGreen - currentGreen);
		int stepsBlue = (int) (endBlue - currentBlue);
		int maxSteps = Math.max(Math.abs(stepsRed), Math.max(Math.abs(stepsGreen), Math.abs(stepsBlue)));

		int[] colors = new int[maxSteps];
		int length = 0;

		// Computes the components progress step.
		float stepRed = (float) stepsRed / maxSteps;
		float stepGreen = (float) stepsGreen / maxSteps;
		float stepBlue = (float) stepsBlue / maxSteps;

		int lastColor = -1;
		while (maxSteps >= 0) {
			// Computes color and saves it if different than the previous one.
			int color = ColorHelper.getColor((int) currentRed, (int) currentGreen, (int) currentBlue);
			int displayColor = display.getDisplayColor(color);
			if (displayColor != lastColor) {
				colors[length] = displayColor;
				length++;
				lastColor = displayColor;
			}
			// Step.
			currentRed += stepRed;
			currentGreen += stepGreen;
			currentBlue += stepBlue;
			maxSteps--;
		}

		// Crops result array to real height.
		int[] result = new int[length];
		System.arraycopy(colors, 0, result, 0, length);
		return result;
	}

}
