/*
 * Copyright 2010-2022 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.nls;

/**
 * NLS provides support for manipulating messages and languages.
 * <p>
 * Each instance of a NLS manages a set of messages, and each message is defined by a unique ID (int value).
 * <p>
 * Each language is defined by a locale string. By convention, a locale represents a specific geographical, political,
 * or cultural region. It is often represented using the concatenation of a language code and a country code:
 * <ul>
 * <li>The language code is the lower-case, two-letter codes as defined by ISO-639.</li>
 * <li>The country code is the upper-case, two-letter codes as defined by ISO-3166.</li>
 * </ul>
 */
public interface NLS {

	/**
	 * Gets the list of available locales.
	 *
	 * @return the list of locales.
	 */
	String[] getAvailableLocales();

	/**
	 * Gets the human-readable name of the given locale.
	 *
	 * @param locale
	 *            the locale.
	 * @return the human-readable name.
	 */
	String getDisplayName(String locale);

	/**
	 * Gets the message matching an ID in the current language.
	 *
	 * @param messageID
	 *            the ID of the message to retrieve.
	 * @return the message at the specified ID.
	 * @throws ArrayIndexOutOfBoundsException
	 *             if the given messageID does not match a valid message.
	 */
	String getMessage(int messageID);

	/**
	 * Gets the message matching an ID in a specific language.
	 *
	 * @param messageID
	 *            the ID of the message to retrieve.
	 * @param locale
	 *            the locale to use.
	 * @return the message at the specified ID.
	 * @throws ArrayIndexOutOfBoundsException
	 *             if the given messageID does not match a valid message.
	 */
	String getMessage(int messageID, String locale);

	/**
	 * Changes the current locale.
	 *
	 * @param locale
	 *            the locale to set.
	 */
	void setCurrentLocale(String locale);

	/**
	 * Gets the current locale.
	 *
	 * @return the current locale.
	 */
	String getCurrentLocale();
}
