/*
 * C
 *
 * Copyright 2014-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 *
 */

#include "LLUI_DISPLAY_impl.h"

#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>		/* open(), O_RDWR */
#include <linux/fb.h>		/* fb_var_screeninfo, ... */
#include <sys/mman.h>		/* mmap() */
#include <string.h>		/* memcopy() */
#include <sys/ioctl.h>
#include <errno.h>
#include <stdlib.h>
#include <stdbool.h>
#include "posix_time.h"
#include "microej.h"
#include "framerate.h"
#include <assert.h>
#include <pthread.h>

// #define LLDISPLAY_DEBUG

#ifdef LLDISPLAY_DEBUG
#define LLDISPLAY_LOG_DEBUG printf
#else // LLDISPLAY_DEBUG
#define LLDISPLAY_LOG_DEBUG(...) ((void) 0)
#endif
#define LLDISPLAY_LOG_WARNING printf("[LLDISPLAY][WARNING] ");printf

//#define DEBUG_SYNC

static uint8_t* back_buffer;

//#define LLDISPLAY_USE_FLIP

#ifdef LLDISPLAY_USE_FLIP
static char* double_buffer1;
static char* double_buffer2;
#endif

static uint8_t* lldisplay_buf = NULL;
static int32_t lldisplay_xmin = 0;
static int32_t lldisplay_xmax = 0;
static int32_t lldisplay_ymin = 0;
static int32_t lldisplay_ymax = 0;

static struct fb_var_screeninfo vscreeninfo; /* copy of drivers var-struct */
static struct fb_fix_screeninfo fscreeninfo; /* copy of drivers fix-struct */
static int fb = -1;	/* file descriptor for the framebuffer device */
static char * fb_base;	/* base address of the video-memory */

static int32_t display_width 	= 0;
static int32_t display_height 	= 0;
static int32_t display_bits_per_pixel 	= 0;
static uint8_t display_convert_32_to_16_bpp = 0;
static uint8_t display_use_vsync = 0;
static uint8_t display_is_available = 1;

static pthread_t lldisplay_copy_threadRef = -1; //interrupt handler pthread

//need to be defined to avoid link issue
int32_t com_ist_allocator_SimpleAllocator_MallocPtr = 0;

typedef struct binary_semaphore_t {
    pthread_mutex_t mutex;
    pthread_cond_t cond;
    bool free;
}binary_semaphore_t;

void lldisplay_binary_semaphore_init(binary_semaphore_t* sem);
void lldisplay_binary_semaphore_take(binary_semaphore_t* sem);
void lldisplay_binary_semaphore_give(binary_semaphore_t* sem);
void lldisplay_mutex_init(pthread_mutex_t* mutex);
void lldisplay_cond_init(pthread_cond_t* condition);

static binary_semaphore_t copy_semaphore;
static binary_semaphore_t binary_semaphore_0;
static binary_semaphore_t binary_semaphore_1;

void lldisplay_binary_semaphore_init(binary_semaphore_t* sem){
	lldisplay_mutex_init(&(sem->mutex));
	lldisplay_cond_init(&(sem->cond));
	sem->free = true;
}

void lldisplay_binary_semaphore_take(binary_semaphore_t* sem)
{
    pthread_mutex_lock(&(sem->mutex));
    while (!sem->free){
        pthread_cond_wait(&(sem->cond), &(sem->mutex));
    }
    sem->free = false;
    pthread_mutex_unlock(&(sem->mutex));
}

void lldisplay_binary_semaphore_give(binary_semaphore_t* sem)
{
	pthread_mutex_lock(&(sem->mutex));
	sem->free = true;
	pthread_cond_signal(&(sem->cond));
	pthread_mutex_unlock(&(sem->mutex));
}

void lldisplay_mutex_init(pthread_mutex_t* mutex){
	pthread_mutexattr_t mutexAttributes;
	int32_t result = pthread_mutexattr_init(&mutexAttributes);
	assert(result==0);

	result = pthread_mutexattr_settype(&mutexAttributes, PTHREAD_MUTEX_DEFAULT);
	assert(result==0);

	result = pthread_mutex_init(mutex, &mutexAttributes);
	assert(result==0);
	result = pthread_mutexattr_destroy(&mutexAttributes);
	assert(result==0);
}

void lldisplay_cond_init(pthread_cond_t* condition){
	// initialize the condition
	pthread_condattr_t conditionAttributes;
	int32_t result = pthread_condattr_init(&conditionAttributes);
	assert(result==0);
#ifndef CONDITION_SETCLOCK_NO_SUPPORT
	// time used by the condition in pthread_cond_timedwait is monotonic
	result = pthread_condattr_setclock(&conditionAttributes, CLOCK_MONOTONIC);
	assert(result==0);
#endif
	result = pthread_cond_init(condition, &conditionAttributes);
	assert(result==0);
	result = pthread_condattr_destroy(&conditionAttributes);
	assert(result==0);
}

static void vsync(void)
{
	if(display_use_vsync){
		int args = 1;
		// wait for vsync
		if(ioctl(fb, FBIO_WAITFORVSYNC, &args)==-1){
			LLDISPLAY_LOG_DEBUG("Framebuffer error during vsync (%s)\n", strerror(errno));
			return ;
		}
	}
}

void* lldisplay_copy_task(void* p_args){
	while(1){
		int32_t  result ;
#ifdef DEBUG_SYNC
		LLDISPLAY_LOG_DEBUG("[TASK] wait copy signal\n");
#endif
		lldisplay_binary_semaphore_take(&copy_semaphore);
#ifdef DEBUG_SYNC
		LLDISPLAY_LOG_DEBUG("[TASK] copy signal got\n");
#endif
#ifdef FRAMERATE_ENABLED
		framerate_increment();
#endif
		vsync();
		int32_t mul = (display_bits_per_pixel/8);
		memcpy((void*)(fb_base+(display_width*lldisplay_ymin*mul)), (void*)(lldisplay_buf+(display_width*lldisplay_ymin*mul)), display_width*(lldisplay_ymax-lldisplay_ymin+1)*mul);

#ifdef DEBUG_SYNC
		LLDISPLAY_LOG_DEBUG("[TASK] give done signal\n");
#endif
		LLUI_DISPLAY_flushDone(false);
#ifdef DEBUG_SYNC
		LLDISPLAY_LOG_DEBUG("[TASK] done signal given\n");
#endif
		assert(result == 0);
	}
}

void LLUI_DISPLAY_IMPL_initialize(LLUI_DISPLAY_SInitData* init_data) {
	char *fb_name = NULL;
	char* display_convert_32_to_16_bpp_str = NULL;
	char* display_use_vsync_str = NULL;
	int screensize = 0;

	LLDISPLAY_LOG_DEBUG("Screen initialization...\n");
	if( (fb_name = getenv("LLDISPLAY_FBDEVICE")) != NULL ) {
		/* open fb-device, map video-memory and more */
		fb = open(fb_name, O_RDWR);
	}else{
		if(!(fb = open("/dev/fb0", O_RDWR))){
			if(!(fb = open("/dev/fb1", O_RDWR))){
				fb = open("/dev/fb", O_RDWR);
			}
		}
	}

	if (fb < 0) {
		LLDISPLAY_LOG_DEBUG("Screen initialization...	FAILED 1\n");
		LLDISPLAY_LOG_WARNING("Frame buffer not available, skipping display setup\n");
		display_is_available = 0;
		return;
	}

	/* get a copy of the current var-structure*/
	if (ioctl(fb, FBIOGET_VSCREENINFO, &vscreeninfo)) {
		LLDISPLAY_LOG_DEBUG("Screen initialization...	FAILED 2\n");
		return;
	}

	display_width = vscreeninfo.xres;
	display_height = vscreeninfo.yres;
	display_bits_per_pixel = vscreeninfo.bits_per_pixel;
	if( (display_convert_32_to_16_bpp_str = getenv("LLDISPLAY_CONVERT_32_TO_16_BPP")) != NULL ) {
		if(display_bits_per_pixel != 16){
			LLDISPLAY_LOG_DEBUG("Wrong screen format (%dx%d - %d bpp) to make convertion of 32 bpp to 16 bpp\n", vscreeninfo.xres, vscreeninfo.yres, display_bits_per_pixel);
			return;
		}
		display_convert_32_to_16_bpp = 1;
	}

	if( (display_use_vsync_str = getenv("LLDISPLAY_USE_VSYNC")) != NULL ) {
		display_use_vsync = 1;
	}

	/* get a copy of the current fix-structure*/
	if (ioctl(fb, FBIOGET_FSCREENINFO, &fscreeninfo)) {
		LLDISPLAY_LOG_DEBUG("Screen initialization...	FAILED 3\n");
		return;
	}

#ifdef LLDISPLAY_USE_FLIP
	/* memory map the frame buffer */
	screensize = vscreeninfo.xres * vscreeninfo.yres * vscreeninfo.bits_per_pixel / 8;
	fb_base = (caddr_t) mmap(0, screensize*2, PROT_READ | PROT_WRITE, MAP_SHARED, fb, 0);
	if (fb_base == (caddr_t) -1) {
		LLDISPLAY_LOG_DEBUG("Screen initialization...	FAILED 4\n");
		return;
	}
	double_buffer1 = fb_base;
	double_buffer2 = fb_base + screensize;

	vscreeninfo.yres_virtual = display_height * 2;
	if(ioctl(fb, FBIOPUT_VSCREENINFO, &vscreeninfo)==-1){
		LLDISPLAY_LOG_DEBUG("Wrong ioctl (FBIOPUT_VSCREENINFO: %s)\n", strerror(errno));
	}

	/* back buffer allocation */
	if((vscreeninfo.bits_per_pixel == 16) || (vscreeninfo.bits_per_pixel == 32)){
		if(display_convert_32_to_16_bpp){
			back_buffer = malloc(vscreeninfo.xres * vscreeninfo.yres * 32 / 8); // 32 bits per pixel stack
		}else{
			back_buffer = malloc(vscreeninfo.xres * vscreeninfo.yres * vscreeninfo.bits_per_pixel / 8);
		}
	}else{
		LLDISPLAY_LOG_DEBUG("Screen format not handle (%dx%d - %d bpp)\n", vscreeninfo.xres, vscreeninfo.yres, vscreeninfo.bits_per_pixel);
		return;
	}
	if(back_buffer == NULL){
		LLDISPLAY_LOG_DEBUG("An error occurred during back buffer allocation\n");
		return;
	}
#else
	/* memory map the frame buffer */
	screensize = vscreeninfo.xres * vscreeninfo.yres * vscreeninfo.bits_per_pixel / 8;
	fb_base = (caddr_t) mmap(0, screensize, PROT_READ | PROT_WRITE, MAP_SHARED, fb, 0);
	if (fb_base == (caddr_t) -1) {
		LLDISPLAY_LOG_DEBUG("Screen initialization...	FAILED 4\n");
		return;
	}

	/* back buffer allocation */
	if((vscreeninfo.bits_per_pixel == 16) || (vscreeninfo.bits_per_pixel == 32)){
		if(display_convert_32_to_16_bpp){
			back_buffer = malloc(vscreeninfo.xres * vscreeninfo.yres * 32 / 8); // 32 bits per pixel stack
		}else{
			back_buffer = malloc(vscreeninfo.xres * vscreeninfo.yres * vscreeninfo.bits_per_pixel / 8);
		}
	}else{
		LLDISPLAY_LOG_DEBUG("Screen format not handle (%dx%d - %d bpp)\n", vscreeninfo.xres, vscreeninfo.yres, vscreeninfo.bits_per_pixel);
		return ;
	}
	if(back_buffer == NULL){
		LLDISPLAY_LOG_DEBUG("An error occurred during back buffer allocation\n");
		return ;
	}
#endif

#ifdef LLDISPLAY_USE_FLIP
	vscreeninfo.xoffset = 0;
	vscreeninfo.yoffset = 0;
	if(ioctl(fb, FBIOPAN_DISPLAY, &vscreeninfo)==-1){
		LLDISPLAY_LOG_DEBUG("Wrong ioctl (FBIOPAN_DISPLAY: %s)\n", strerror(errno));
	}
#endif

	// initialize copy task semaphore
	lldisplay_binary_semaphore_init(&copy_semaphore);
	//take the semaphore => next take should block
	lldisplay_binary_semaphore_take(&copy_semaphore);

	int32_t result ;
	pthread_attr_t attributes;
	result = pthread_attr_init(&attributes);
	assert(result==0);
	result = pthread_attr_setstacksize(&attributes, PTHREAD_STACK_MIN);
	assert(result==0);
	// Initialize pthread such as its resource will be
	result = pthread_attr_setdetachstate(&attributes, PTHREAD_CREATE_JOINABLE);
	result = pthread_create(&lldisplay_copy_threadRef, &attributes, &lldisplay_copy_task, NULL);
	assert(result==0);

	lldisplay_binary_semaphore_init(&binary_semaphore_0);
	lldisplay_binary_semaphore_init(&binary_semaphore_1);
	init_data->binary_semaphore_0 = (void*)&binary_semaphore_0;
	init_data->binary_semaphore_1 = (void*)&binary_semaphore_1;
	init_data->lcd_width = display_width;
	init_data->lcd_height = display_height;
	init_data->back_buffer_address = (uint8_t*)back_buffer;
	LLDISPLAY_LOG_DEBUG("Screen initialization...	OK\n");
}

uint8_t* LLUI_DISPLAY_IMPL_flush(MICROUI_GraphicsContext* gc, uint8_t* addr, uint32_t xmin, uint32_t ymin, uint32_t xmax, uint32_t ymax)
{

	(void)gc;
	lldisplay_buf = addr;
	lldisplay_ymin = ymin;
	lldisplay_ymax = ymax;

	if (!display_is_available) {
		return addr;
	}

#ifdef DEBUG_SYNC
		LLDISPLAY_LOG_DEBUG("[FLUSH] give copy signal\n");
#endif
	lldisplay_binary_semaphore_give(&copy_semaphore);
#ifdef DEBUG_SYNC
		LLDISPLAY_LOG_DEBUG("[FLUSH] copy signal given\n");
#endif

#ifdef FRAMERATE_ENABLED
//	framerate_increment();
#endif

	return addr;
}

void LLUI_DISPLAY_IMPL_binarySemaphoreTake(void* sem)
{
	if (!display_is_available) {
		return;
	}
	lldisplay_binary_semaphore_take((binary_semaphore_t*)sem);
}

void LLUI_DISPLAY_IMPL_binarySemaphoreGive(void* sem, bool under_isr)
{
	(void)under_isr;
	if (!display_is_available) {
		return;
	}
	lldisplay_binary_semaphore_give((binary_semaphore_t*)sem);
}
