/*
 * Java
 *
 * Copyright 2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.event;

/**
 * Thrown to indicate that a method has caused an internal EventQueue error.
 * This error is specified by an identifier
 */

public class EventQueueException extends RuntimeException {
    private EventQueueException() {}

    /**
     * Error code <code>-1</code>: thrown when an invalid event type is provided.
     * <p>
     * This occurs if the event type is lower than <code>0</code> or greater than <code>127</code>
     *
     * <p>
     * Typically thrown by {@link ej.event.EventQueue#offerEvent(int, int)} or
     * {@link ej.event.EventQueue#offerExtendedEvent(int type, byte[])}.
     */
    public static final int INVALID_TYPE_MUST_BE_BETWEEN_0_AND_127 = -1;

    /**
     * Error code <code>-2</code>: thrown when the FIFO buffer is full.
     * <p>
     * Indicates that the queue cannot accept any new events until
     * previously queued events are processed.
     */
    public static final int FIFO_IS_FULL = -2;

    /**
     * Error code <code>-3</code>: thrown when the data value exceeds 24 bits.
     * <p>
     * This can occur when using {@link ej.event.EventQueue#offerEvent(int, int)} with an
     * {@code data} that cannot be represented within 24 bits, or when calling
     * {@link ej.event.EventQueue#offerExtendedEvent(int, byte[])} with a {@code data} buffer
     * whose length exceeds 24 bits.
     */
    public static final int DATA_EXCEED_24_BIT = -3;



    /**
     * Error code <code>-4</code>: thrown when the maximum number of event types
     * has already been registered.
     */
    public static final int MAXIMUM_REGISTERED_EVENTS_REACHED = -4;

    /**
     * Error code <code>-5</code>: thrown when no listener is registered for the specified event type.
     * <p>
     * This occurs when an event is offered with a type value that does not correspond to any registered listener.
     * <p>
     *
     * @see ej.event.EventQueue#registerListener(ej.event.EventQueueListener)
     * @see ej.event.EventQueue#unregisterListener(int)
     */
    public static final int NO_LISTENER_REGISTERED = -5;

    /**
     * Error code <code>-6</code>: thrown when attempting to set the default listener to {@code null}.
     * <p>
     * This occurs when calling {@link ej.event.EventQueue#setDefaultListener(ej.event.EventQueueListener)}
     * with a {@code null} reference. The default listener is required to handle events that do not
     * correspond to any registered listener type.
     * <p>
     *
     * @see ej.event.EventQueue#setDefaultListener(ej.event.EventQueueListener)
     */
    public static final int DEFAULT_LISTENER_NULL = -6;

    /**
     * Error code <code>-7</code>: thrown if there is not enough bytes remaining in the extended event.
     *  <p>
     * This occurs when calling {@link ej.event.EventQueueDataReader#readFully(byte[])}
     */
    public static final int NOT_ENOUGH_BYTES_REMAINING = -7;
}
