/*
 * Copyright 2015-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.style.outline.border;

import ej.annotation.Nullable;
import ej.bon.XMath;
import ej.microui.display.GraphicsContext;
import ej.mwt.style.outline.Outline;
import ej.mwt.util.Outlineable;
import ej.mwt.util.Size;

/**
 * Represents and draw a rectangular colored border.
 * <p>
 * The thickness is stored as a <code>char</code> for heap optimization and therefore cannot exceed <code>65535</code>.
 */
public class RectangularBorder implements Outline {

	private final int color;
	private final char thickness;

	/**
	 * Creates a rectangular border specifying its thickness.
	 * <p>
	 * The given thickness is clamped between <code>0</code> and <code>Character.MAX_VALUE</code>.
	 *
	 * @param color
	 *            the color.
	 * @param thickness
	 *            the thickness.
	 */
	public RectangularBorder(int color, int thickness) {
		this.color = color;
		this.thickness = (char) XMath.limit(thickness, 0, Character.MAX_VALUE);
	}

	/**
	 * Gets the color.
	 *
	 * @return the color.
	 */
	public int getColor() {
		return this.color;
	}

	/**
	 * Gets the thickness.
	 *
	 * @return the thickness.
	 */
	public int getThickness() {
		return this.thickness;
	}

	@Override
	public void apply(Outlineable outlineable) {
		int thickness = this.thickness;
		outlineable.removeOutline(thickness, thickness, thickness, thickness);
	}

	@Override
	public void apply(GraphicsContext g, Size size) {
		int width = size.getWidth();
		int height = size.getHeight();

		int thickness = this.thickness;

		g.setColor(this.color);
		BorderHelper.drawRectangularBorder(g, width, height, thickness, thickness, thickness, thickness);

		size.removeOutline(thickness, thickness, thickness, thickness);

		g.translate(thickness, thickness);
		g.intersectClip(0, 0, size.getWidth(), size.getHeight());
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		if (obj instanceof RectangularBorder) {
			RectangularBorder border = (RectangularBorder) obj;
			return this.color == border.color && this.thickness == border.thickness;
		}
		return false;
	}

	@Override
	public int hashCode() {
		return 11 * this.color + 13 * this.thickness;
	}

}
