/*
 * Java
 *
 * Copyright 2019-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.event;

import java.util.HashMap;
import java.util.Map;

import ej.fp.FrontPanel;

/**
 * This class holds all methods to send MicroUI States events to MicroUI application.
 * <p>
 * Contrary to the other event generators, the MicroUI States generator must be initialized with the default state of
 * each state. MicroUI implementation calls the method {@link #getInitialStateValue(int, int)} on startup (after front
 * panel device startup). The front panel project has the responsability to call the method
 * {@link #setInitialStateValue(int, int, int)} before device starting in order to fill the states.
 */
@SuppressWarnings("nls")
public class EventState {

	/**
	 * Usual MicroUI States generator tag used in platform configuration project (<code>microui.xml</code>).
	 */
	public static final String COMMON_MICROUI_GENERATOR_TAG = "STATES";

	// MicroUI states events representation
	private static final int STATES_DRIVER_VALUE_MASK = 0x0000ff00;
	private static final int STATES_DRIVER_VALUE_SHIFT = 8;
	private static final int STATES_DRIVER_STATE_MASK = 0x000000ff;
	private static final int STATES_DRIVER_STATE_SHIFT = 0;

	private static final int DEFAULT_INITIAL_VALUE = 0;

	private static final HashMap<Integer, HashMap<Integer, Integer>> EventGeneratorIDtoState = new HashMap<>(0);

	private EventState() {
		// hide implicit constructor
	}

	/**
	 * Send a STATE event to the event generator with the MicroUI generator tag {@link #COMMON_MICROUI_GENERATOR_TAG}.
	 *
	 * @param stateID
	 *            the state machine identifier, between 0 and 255.
	 * @param stateValue
	 *            the new state value, between 0 and 255.
	 *
	 * @see #sendStateEvent(String, int, int)
	 */
	public static void sendStateEvent(int stateID, int stateValue) {
		sendStateEvent(COMMON_MICROUI_GENERATOR_TAG, stateID, stateValue);
	}

	/**
	 * Send a STATE event to the event generator.
	 * <p>
	 * The result of this method is unknown when the given event generator is not a MicroUI <code>States</code> event
	 * generator.
	 *
	 * @param statesEventGenerator
	 *            the MicroUI <code>States</code> event generator.
	 * @param stateID
	 *            the state machine identifier, between 0 and 255.
	 * @param stateValue
	 *            the new state value, between 0 and 255.
	 *
	 */
	public static void sendStateEvent(String statesEventGenerator, int stateID, int stateValue) {
		try {
			LLUIInput.sendEvent(statesEventGenerator,
					((stateID << STATES_DRIVER_STATE_SHIFT) & STATES_DRIVER_STATE_MASK)
							| ((stateValue << STATES_DRIVER_VALUE_SHIFT) & STATES_DRIVER_VALUE_MASK));
		} catch (IllegalArgumentException e) {
			// invalid MicroUI events generator ID
			// -> print a trace
			StringBuilder buf = new StringBuilder();
			buf.append("[");
			buf.append(EventState.class.getName());
			buf.append("] ");
			buf.append("state ");
			buf.append(stateID);
			buf.append("value ");
			buf.append(stateValue);
			buf.append(" (");
			buf.append(statesEventGenerator);
			buf.append(")");
			FrontPanel.out.println(buf.toString());
		}
	}

	/**
	 * Initializes a state generator with its default state on startup.
	 *
	 * @param eventGeneratorID
	 *            the MicroUI <code>States</code> event generator.
	 * @param stateID
	 *            the state machine identifier, between 0 and 255.
	 * @param value
	 *            the state value, between 0 and 255.
	 */
	public static void setInitialStateValue(int eventGeneratorID, int stateID, int value) {

		synchronized (EventGeneratorIDtoState) {
			// create objects from method arguments
			Integer iEventGeneratorID = new Integer(eventGeneratorID);
			Integer iStateID = new Integer(stateID);
			Integer iValue = new Integer(value);

			// retrieve the event generator states if exists
			HashMap<Integer, Integer> state = EventGeneratorIDtoState.get(iEventGeneratorID);

			if (state == null) {
				// create new event generator states for this event generator ID
				state = new HashMap<Integer, Integer>(1);
				EventGeneratorIDtoState.put(iEventGeneratorID, state);
			}

			// store state value for current event generator
			state.put(iStateID, iValue);
		}
	}

	/**
	 * Gets the state initial value (startup value). Called by MicroUI implementation on startup.
	 *
	 * @param eventGeneratorID
	 *            the MicroUI <code>States</code> event generator.
	 * @param stateID
	 *            the state machine identifier, between 0 and 255.
	 * @return state initial value.
	 */
	public static int getInitialStateValue(int eventGeneratorID, int stateID) {

		synchronized (EventGeneratorIDtoState) {
			// create object from method argument
			Integer iStateID = new Integer(stateID);

			int stateValue = DEFAULT_INITIAL_VALUE;

			// look for states event generator
			for (Map.Entry<Integer, HashMap<Integer, Integer>> entry : EventGeneratorIDtoState.entrySet()) {

				Integer gen = entry.getKey();
				if (gen.intValue() == eventGeneratorID) {

					// states event generator found, retrieve its states
					HashMap<Integer, Integer> state = EventGeneratorIDtoState.get(gen);

					// retrieve state value from event generator states
					Integer iValue = state.get(iStateID);
					if (iValue != null) {
						stateValue = iValue.intValue();
					}
					// else: no initial value set for this state, return default value

					// states event generator found, stop research
					break;
				}
			}

			return stateValue;
		}
	}
}
