/*
 * Java
 *
 * Copyright 2019-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.library.iot.rcommand.bluetooth;

import ej.bluetooth.BluetoothAddress;
import ej.bluetooth.BluetoothAttribute;
import ej.bluetooth.BluetoothCharacteristic;
import ej.bluetooth.BluetoothConnection;
import ej.bluetooth.BluetoothDescriptor;
import ej.bluetooth.BluetoothService;
import ej.bluetooth.listeners.ConnectionListener;
import ej.bluetooth.listeners.LocalServiceListener;
import ej.bluetooth.listeners.RemoteServiceListener;
import ej.bon.ByteArray;

public class EventSerializer implements ConnectionListener, RemoteServiceListener, LocalServiceListener {

	private static final byte ADDRESS_TYPE_PUBLIC = 0x00;
	private static final byte ADDRESS_TYPE_PRIVATE = 0x01;

	private static final byte ATTRIBUTE_TYPE_CHARACTERISTIC = 0;
	private static final byte ATTRIBUTE_TYPE_DESCRIPTOR = 1;

	private static final int SERVICE_HEADER_SIZE = 22;
	private static final int ATTRIBUTE_SIZE = 22;

	private final BluetoothController controller;

	public EventSerializer(BluetoothController controller) {
		this.controller = controller;
	}

	@Override
	public void onScanResult(BluetoothAddress address, byte[] advertisementData, int rssi) {
		byte[] event = new byte[10 + advertisementData.length];
		event[0] = BluetoothEventTypes.GAP_SCAN_RESULT;
		event[1] = (address.isPublic() ? ADDRESS_TYPE_PUBLIC : ADDRESS_TYPE_PRIVATE);
		address.getBytes(event, 2);
		event[8] = (byte) rssi;
		event[9] = (byte) advertisementData.length;
		System.arraycopy(advertisementData, 0, event, 10, advertisementData.length);
		this.controller.sendEvent(event);
	}

	@Override
	public void onScanCompleted() {
		byte[] event = new byte[1];
		event[0] = BluetoothEventTypes.GAP_SCAN_COMPLETED;
		this.controller.sendEvent(event);
	}

	@Override
	public void onAdvertisementCompleted() {
		byte[] event = new byte[1];
		event[0] = BluetoothEventTypes.GAP_ADVERTISEMENT_COMPLETED;
		this.controller.sendEvent(event);
	}

	@Override
	public void onConnectFailed(BluetoothAddress address) {
		byte[] event = new byte[8];
		event[0] = BluetoothEventTypes.GAP_CONNECT_FAILED;
		event[1] = (address.isPublic() ? ADDRESS_TYPE_PUBLIC : ADDRESS_TYPE_PRIVATE);
		address.getBytes(event, 2);
		this.controller.sendEvent(event);
	}

	@Override
	public void onConnected(BluetoothConnection connection) {
		BluetoothAddress address = connection.getAddress();
		short connHandle = this.controller.onConnected(connection);

		byte[] event = new byte[10];
		event[0] = BluetoothEventTypes.GAP_CONNECTED;
		event[1] = (address.isPublic() ? ADDRESS_TYPE_PUBLIC : ADDRESS_TYPE_PRIVATE);
		address.getBytes(event, 2);
		ByteArray.writeShort(event, 8, connHandle);
		this.controller.sendEvent(event);
	}

	@Override
	public void onDisconnected(BluetoothConnection connection) {
		Short connHandle = this.controller.onDisconnected(connection);
		if (connHandle != null) {
			byte[] event = new byte[4];
			event[0] = BluetoothEventTypes.GAP_DISCONNECTED;
			ByteArray.writeShort(event, 2, connHandle.shortValue());
			this.controller.sendEvent(event);
		}
	}

	@Override
	public void onPairRequest(BluetoothConnection connection) {
		Short connHandle = this.controller.getConnHandle(connection);
		if (connHandle != null) {
			byte[] event = new byte[4];
			event[0] = BluetoothEventTypes.GAP_PAIR_REQUEST;
			ByteArray.writeShort(event, 2, connHandle.shortValue());
			this.controller.sendEvent(event);
		}
	}

	@Override
	public void onPairCompleted(BluetoothConnection connection, boolean success) {
		Short connHandle = this.controller.getConnHandle(connection);
		if (connHandle != null) {
			byte[] event = new byte[4];
			event[0] = BluetoothEventTypes.GAP_PAIR_COMPLETED;
			event[1] = (byte) (success ? 1 : 0);
			ByteArray.writeShort(event, 2, connHandle.shortValue());
			this.controller.sendEvent(event);
		}
	}

	@Override
	public void onPasskeyRequest(BluetoothConnection connection) {
		Short connHandle = this.controller.getConnHandle(connection);
		if (connHandle != null) {
			byte[] event = new byte[4];
			event[0] = BluetoothEventTypes.GAP_PASSKEY_REQUEST;
			ByteArray.writeShort(event, 2, connHandle.shortValue());
			this.controller.sendEvent(event);
		}
	}

	@Override
	public void onPasskeyGenerated(BluetoothConnection connection, int passkey) {
		Short connHandle = this.controller.getConnHandle(connection);
		if (connHandle != null) {
			byte[] event = new byte[8];
			event[0] = BluetoothEventTypes.GAP_PASSKEY_GENERATED;
			ByteArray.writeShort(event, 2, connHandle.shortValue());
			ByteArray.writeInt(event, 4, passkey);
			this.controller.sendEvent(event);
		}
	}

	@Override
	public void onDiscoveryResult(BluetoothConnection connection, BluetoothService service) {
		Short connHandle = this.controller.getConnHandle(connection);
		if (connHandle != null) {
			Short serviceHandle = this.controller.onServiceDiscovered(connection, service);
			if (serviceHandle != null) {
				int numAttributes = countAttributes(service);
				byte[] event = new byte[SERVICE_HEADER_SIZE + numAttributes * ATTRIBUTE_SIZE];
				event[0] = BluetoothEventTypes.GATTC_DISCOVERY_RESULT;

				// write conn handle
				ByteArray.writeShort(event, 2, connHandle.shortValue());

				// write service data
				event[1] = (byte) numAttributes;
				ByteArray.writeShort(event, 4, serviceHandle.shortValue());
				service.getUuid().getBytes(event, 6);

				int attrOffset = SERVICE_HEADER_SIZE;
				int numCharacteristics = service.getNumCharacteristics();
				for (int c = 0; c < numCharacteristics; c++) {
					// write characteristic data
					BluetoothCharacteristic characteristic = service.getCharacteristic(c);
					Short characteristicHandle = this.controller.getRemoteAttributeHandle(connection, characteristic);
					if (characteristicHandle == null) {
						return;
					}
					event[attrOffset] = ATTRIBUTE_TYPE_CHARACTERISTIC;
					characteristic.getUuid().getBytes(event, attrOffset + 2);
					ByteArray.writeShort(event, attrOffset + 18, characteristicHandle.shortValue());
					event[attrOffset + 20] = characteristic.getProperties();
					attrOffset += ATTRIBUTE_SIZE;

					int numDescriptors = characteristic.getNumDescriptors();
					for (int d = 0; d < numDescriptors; d++) {
						// write descriptor data
						BluetoothDescriptor descriptor = characteristic.getDescriptor(d);
						Short descriptorHandle = this.controller.getRemoteAttributeHandle(connection, descriptor);
						if (descriptorHandle == null) {
							return;
						}
						event[attrOffset] = ATTRIBUTE_TYPE_DESCRIPTOR;
						descriptor.getUuid().getBytes(event, attrOffset + 2);
						ByteArray.writeShort(event, attrOffset + 18, descriptorHandle.shortValue());
						attrOffset += ATTRIBUTE_SIZE;
					}
				}

				this.controller.sendEvent(event);
			}
		}
	}

	@Override
	public void onDiscoveryCompleted(BluetoothConnection connection) {
		Short connHandle = this.controller.getConnHandle(connection);
		if (connHandle != null) {
			byte[] event = new byte[4];
			event[0] = BluetoothEventTypes.GATTC_DISCOVERY_COMPLETED;
			ByteArray.writeShort(event, 2, connHandle.shortValue());
			this.controller.sendEvent(event);
		}
	}

	@Override
	public void onReadCompleted(BluetoothConnection connection, BluetoothAttribute attribute, byte status,
			byte[] value) {
		Short connHandle = this.controller.getConnHandle(connection);
		if (connHandle != null) {
			Short attrHandle = this.controller.getRemoteAttributeHandle(connection, attribute);
			if (attrHandle != null) {
				byte[] event = new byte[8 + value.length];
				event[0] = BluetoothEventTypes.GATTC_READ_COMPLETED;
				event[1] = status;
				ByteArray.writeShort(event, 2, connHandle.shortValue());
				ByteArray.writeShort(event, 4, attrHandle.shortValue());
				ByteArray.writeShort(event, 6, value.length);
				System.arraycopy(value, 0, event, 8, value.length);
				this.controller.sendEvent(event);
			}
		}
	}

	@Override
	public void onWriteCompleted(BluetoothConnection connection, BluetoothAttribute attribute, byte status) {
		Short connHandle = this.controller.getConnHandle(connection);
		if (connHandle != null) {
			Short attrHandle = this.controller.getRemoteAttributeHandle(connection, attribute);
			if (attrHandle != null) {
				byte[] event = new byte[6];
				event[0] = BluetoothEventTypes.GATTC_WRITE_COMPLETED;
				event[1] = status;
				ByteArray.writeShort(event, 2, connHandle.shortValue());
				ByteArray.writeShort(event, 4, attrHandle.shortValue());
				this.controller.sendEvent(event);
			}
		}
	}

	@Override
	public void onNotificationReceived(BluetoothConnection connection, BluetoothCharacteristic characteristic,
			byte[] value) {
		Short connHandle = this.controller.getConnHandle(connection);
		if (connHandle != null) {
			Short attrHandle = this.controller.getRemoteAttributeHandle(connection, characteristic);
			if (attrHandle != null) {
				byte[] event = new byte[8 + value.length];
				event[0] = BluetoothEventTypes.GATTC_NOTIFICATION_RECEIVED;
				ByteArray.writeShort(event, 2, connHandle.shortValue());
				ByteArray.writeShort(event, 4, attrHandle.shortValue());
				ByteArray.writeShort(event, 6, value.length);
				System.arraycopy(value, 0, event, 8, value.length);
				this.controller.sendEvent(event);
			}
		}
	}

	@Override
	public void onReadRequest(BluetoothConnection connection, BluetoothAttribute attribute) {
		onReadBlobRequest(connection, attribute, 0);
	}

	@Override
	public void onReadBlobRequest(BluetoothConnection connection, BluetoothAttribute attribute, int offset) {
		Short connHandle = this.controller.getConnHandle(connection);
		if (connHandle != null) {
			Short attrHandle = this.controller.getLocalAttributeHandle(attribute);
			if (attrHandle != null) {
				byte[] event = new byte[8];
				event[0] = BluetoothEventTypes.GATTS_READ_REQUEST;
				ByteArray.writeShort(event, 2, connHandle.shortValue());
				ByteArray.writeShort(event, 4, attrHandle.shortValue());
				ByteArray.writeShort(event, 6, offset);
				this.controller.sendEvent(event);
			}
		}
	}

	@Override
	public void onWriteRequest(BluetoothConnection connection, BluetoothAttribute attribute, byte[] value) {
		onWriteRequest(connection, attribute, value, 0, false);
	}

	@Override
	public void onPrepareWriteRequest(BluetoothConnection connection, BluetoothAttribute attribute, byte[] value,
			int offset) {
		onWriteRequest(connection, attribute, value, offset, true);
	}

	@Override
	public void onExecuteWriteRequest(BluetoothConnection connection, BluetoothAttribute attribute, boolean execute) {
		Short connHandle = this.controller.getConnHandle(connection);
		if (connHandle != null) {
			Short attrHandle = this.controller.getLocalAttributeHandle(attribute);
			if (attrHandle != null) {
				byte[] event = new byte[6];
				event[0] = BluetoothEventTypes.GATTS_EXECUTE_WRITE_REQUEST;
				event[1] = (byte) (execute ? 1 : 0);
				ByteArray.writeShort(event, 2, connHandle.shortValue());
				ByteArray.writeShort(event, 4, attrHandle.shortValue());
				this.controller.sendEvent(event);
			}
		}
	}

	@Override
	public void onNotificationSent(BluetoothConnection connection, BluetoothCharacteristic characteristic,
			boolean success) {
		Short connHandle = this.controller.getConnHandle(connection);
		if (connHandle != null) {
			Short attrHandle = this.controller.getLocalAttributeHandle(characteristic);
			if (attrHandle != null) {
				byte[] event = new byte[6];
				event[0] = BluetoothEventTypes.GATTS_NOTIFICATION_SENT;
				event[1] = (byte) (success ? 1 : 0);
				ByteArray.writeShort(event, 2, connHandle.shortValue());
				ByteArray.writeShort(event, 4, attrHandle.shortValue());
				this.controller.sendEvent(event);
			}
		}
	}

	private void onWriteRequest(BluetoothConnection connection, BluetoothAttribute attribute, byte[] value, int offset,
			boolean prepare) {
		Short connHandle = this.controller.getConnHandle(connection);
		if (connHandle != null) {
			Short attrHandle = this.controller.getLocalAttributeHandle(attribute);
			if (attrHandle != null) {
				byte[] event = new byte[10 + value.length];
				event[0] = BluetoothEventTypes.GATTS_WRITE_REQUEST;
				event[1] = (byte) (prepare ? 1 : 0);
				ByteArray.writeShort(event, 2, connHandle.shortValue());
				ByteArray.writeShort(event, 4, attrHandle.shortValue());
				ByteArray.writeShort(event, 6, offset);
				ByteArray.writeShort(event, 8, value.length);
				System.arraycopy(value, 0, event, 10, value.length);
				this.controller.sendEvent(event);
			}
		}
	}

	private static int countAttributes(BluetoothService service) {
		int numCharacteristics = service.getNumCharacteristics();
		int numDescriptors = 0;
		for (int c = 0; c < numCharacteristics; c++) {
			numDescriptors += service.getCharacteristic(c).getNumDescriptors();
		}
		return numCharacteristics + numDescriptors;
	}
}
