/*
 * Copyright 2010-2015 http://beders.github.com/Resty
 * Copyright 2015-2019 MicroEJ Corp. This file has been modified by MicroEJ Corp.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ej.rest.web.mime;

import java.io.IOException;
import java.io.OutputStream;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Random;

import ej.rest.web.AbstractContent;


/** Content handler for multipart data of any shape (form, mixed, alternative)
 * 
 * @author beders
 *
 */
public class MultipartContent extends AbstractContent {
	private final String subType;
	private final List<AbstractContent> parts;
	private final String boundary = "jb" + randomUUID();

	public MultipartContent(String aSubtype, AbstractContent... content) {
		subType = aSubtype;
		parts = new ArrayList<AbstractContent>(Arrays.asList(content));
	}

	@Override
	protected void addContent(URLConnection con) throws IOException {
		con.setDoOutput(true);
		con.addRequestProperty("Content-Type", "multipart/" + subType + "; boundary=" + boundary);
		OutputStream os = con.getOutputStream();
		writeContent(os);
		os.close();
	}

	@Override
	public void writeContent(OutputStream os) throws IOException {
		if (parts.isEmpty()) {
			return;
		}
		for (AbstractContent c : parts) {
			os.write(ascii("--"));
			os.write(ascii(boundary));
			os.write(CRLF);
			c.writeHeader(os);
			os.write(CRLF);
			c.writeContent(os);
			os.write(CRLF);
		}
		os.write(ascii("--"));
		os.write(ascii(boundary));
		os.write(ascii("--"));
		os.write(CRLF);
	}


	@Override
	public void writeHeader(OutputStream os) throws IOException {
		os.write(ascii("Content-Type: multipart/" + subType + "; boundary=" + boundary + "\r\n"));
	}

	private static String randomUUID() {

		Random ng = new Random();

		byte[] randomBytes = new byte[16];

		ng.nextBytes(randomBytes);
		randomBytes[6]  &= 0x0f;  /* clear version        */
		randomBytes[6]  |= 0x40;  /* set to version 4     */
		randomBytes[8]  &= 0x3f;  /* clear variant        */
		randomBytes[8]  |= 0x80;  /* set to IETF variant  */

		long msb = 0;
		long lsb = 0;

		for (int i=0; i<8; i++){
			msb = (msb << 8) | (randomBytes[i] & 0xff);
		}

		for (int i=8; i<16; i++){
			lsb = (lsb << 8) | (randomBytes[i] & 0xff);
		}

		return uuidtoString(msb, lsb);

	}

	private static String uuidtoString(long mostSigBits, long leastSigBits) {

		return (digits(mostSigBits >> 32, 8) + "-" +
				digits(mostSigBits >> 16, 4) + "-" +
				digits(mostSigBits, 4) + "-" +
				digits(leastSigBits >> 48, 4) + "-" +
				digits(leastSigBits, 12));

	}

	private static String digits(long val, int digits) {
		long hi = 1L << (digits * 4);
		return Long.toHexString(hi | (val & (hi - 1))).substring(1);

	}

}
