/*
 * Java
 *
 * Copyright 2018-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.util.message.basic;

import ej.util.message.MessageBuilder;
import ej.util.message.MessageLogger;

/**
 * A basic message logger delegates the messages build to a {@link MessageBuilder} and prints them to the standard
 * output stream.
 *
 * @see System#out
 */
public class BasicMessageLogger implements MessageLogger {

	private MessageBuilder builder;

	/**
	 * An instance of BasicMessageLogger.
	 */
	public static final BasicMessageLogger INSTANCE = new BasicMessageLogger();

	/**
	 * Creates an instance using {@link BasicMessageBuilder#INSTANCE} to build the messages.
	 */
	public BasicMessageLogger() {
		this(BasicMessageBuilder.INSTANCE);
	}

	/**
	 * Creates an instance passing the message builder used to build the messages.
	 *
	 * @param builder
	 *            the {@link MessageBuilder} to use, cannot be <code>null</code>.
	 * @throws NullPointerException
	 *             if the given builder is <code>null</code>.
	 */
	public BasicMessageLogger(MessageBuilder builder) {
		if (builder == null) {
			throw new NullPointerException();
		}
		this.builder = builder;
	}

	@Override
	public MessageBuilder getMessageBuilder() {
		return this.builder;
	}

	@Override
	public void setMessageBuilder(MessageBuilder builder) {
		if (builder == null) {
			throw new NullPointerException();
		}
		this.builder = builder;
	}

	@Override
	public void log(char level, String category, int id) {
		System.out.println(this.builder.buildMessage(level, category, id));
	}

	@Override
	public void log(char level, String category, int id, Throwable t) {
		System.out.println(this.builder.buildMessage(level, category, id, t));
	}

	@Override
	public void log(char level, String category, int id, Object... arguments) {
		System.out.println(this.builder.buildMessage(level, category, id, arguments));

	}

	@Override
	public void log(char level, String category, int id, Throwable t, Object... arguments) {
		System.out.println(this.builder.buildMessage(level, category, id, t, arguments));
	}

}
