/*
 * Java
 *
 * Copyright 2015-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.net;

import java.io.IOException;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.Socket;
import java.net.SocketException;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;

import android.net.ConnectivityManager;
import android.net.NetworkCapabilities;
import ej.bon.Timer;
import ej.bon.TimerTask;
import ej.net.util.NetUtil;

/**
 * Basic full java implementation. Consider the network is connected when there is at least one {@link NetworkInterface}
 * which address is not <code>0.0.0.0</code>.
 */
public class PollerConnectivityManager extends ConnectivityManager {

	/** package **/
	static final NetworkCapabilities CONNECTED = new NetworkCapabilities(
			new int[] { NetworkCapabilities.NET_CAPABILITY_INTERNET });
	/** package **/
	static final NetworkCapabilities UNCONNECTED = new NetworkCapabilities();

	private static final boolean DEBUG = false;
	private final Timer timer;
	private final Map<String, Network> networks;

	private static final String BASE_PROPERTY = "connectivity."; //$NON-NLS-1$
	/**
	 * Default url used for testing internet connectivity.
	 */
	private static final String CONNECT = "connect."; //$NON-NLS-1$
	private static final String DEFAULT_URL = System.getProperty(BASE_PROPERTY + CONNECT + "url", "microej.com"); //$NON-NLS-1$ //$NON-NLS-2$
	private static final int DEFAULT_PORT = Integer.getInteger(BASE_PROPERTY + CONNECT + "port", 443).intValue(); //$NON-NLS-1$

	private static final String TASK = "task."; //$NON-NLS-1$
	/**
	 * Default delay is 0 : means start polling ASAP
	 */
	private static final long DEFAULT_DELAY = Long.getLong(BASE_PROPERTY + TASK + "delay", 0).longValue(); //$NON-NLS-1$

	/**
	 * Default period when there's no connectivity.
	 * <p>
	 * Likely to be shorter than the {@link DEFAULT_PERIOD_WHEN_AVAILABLE} as we want to detect the network availability
	 * as soon as possible.
	 */
	private static final long DEFAULT_PERIOD_WHEN_LOST = Long.getLong(BASE_PROPERTY + TASK + "period", 500).longValue(); //$NON-NLS-1$

	/**
	 * Default period when there's connectivity.
	 * <p>
	 * Likely to be longer than the {@link DEFAULT_PERIOD_WHEN_LOST}, as it's likely that the network will not go down,
	 * and it's acceptable not to detect it immediately.
	 */
	private static final long DEFAULT_PERIOD_WHEN_AVAILABLE = Long
			.getLong(BASE_PROPERTY + TASK + "period_when_up", 10 * DEFAULT_PERIOD_WHEN_LOST).longValue(); //$NON-NLS-1$

	/**
	 * Default period when there is internet connectivity.
	 * <p>
	 * Likely to be longer than the {@link DEFAULT_PERIOD_WHEN_AVAILABLE}, as it's likely that the internet will not go
	 * down, and it's acceptable not to detect it immediately.
	 */
	private static final long DEFAULT_PERIOD_WHEN_INTERNET = Long
			.getLong(BASE_PROPERTY + TASK + "period_when_internet", DEFAULT_PERIOD_WHEN_AVAILABLE).longValue(); //$NON-NLS-1$
	private static final String EMPTY_NETWORK = ""; //$NON-NLS-1$

	private final long periodWhenLost;
	private final long periodWhenAvailable;
	private final long periodWhenInternet;
	private final String url;
	private final int port;
	private ConnectivityManagerTimerTask pollerTask;
	private Network activeNetwork;
	private long currentPeriod;

	/**
	 * Instantiates a {@link PollerConnectivityManager} with default periods and delay and a new timer.
	 */
	public PollerConnectivityManager() {
		this(new Timer());
		timer.schedule(new TimerTask() {

			@Override
			public void run() {
				Thread.currentThread().setName(PollerConnectivityManager.class.getSimpleName());
			}
		}, 0);
	}

	/**
	 * Instantiates a {@link PollerConnectivityManager} with default value.
	 *
	 * @param timer
	 *            the timer to use.
	 */
	public PollerConnectivityManager(Timer timer) {
		this(timer, DEFAULT_DELAY, DEFAULT_PERIOD_WHEN_AVAILABLE, DEFAULT_PERIOD_WHEN_LOST,
				DEFAULT_PERIOD_WHEN_INTERNET);
	}

	/**
	 * Instantiates a {@link PollerConnectivityManager} with the same period when up or not.
	 *
	 * @param timer
	 *            the timer to use.
	 * @param delay
	 *            the delay before the start.
	 * @param period
	 *            the period for the polling.
	 */
	public PollerConnectivityManager(Timer timer, long delay, long period) {
		this(timer, delay, period, period);
	}

	/**
	 * Instantiates a {@link PollerConnectivityManager} with the different periods when available or not. The same period is used wth or without internet access when availble.
	 *
	 * @param timer
	 *            the timer to use.
	 * @param delay
	 *            the delay before the start.
	 * @param periodWhenAvailable
	 *            the period for the polling when at least one interface is available.
	 * @param periodWhenLost
	 *            the period for the polling when all the interface are down.
	 */
	public PollerConnectivityManager(Timer timer, long delay, long periodWhenAvailable, long periodWhenLost) {
		this(timer, delay, periodWhenAvailable, periodWhenLost, periodWhenAvailable);
	}

	/**
	 * Instantiates a {@link PollerConnectivityManager} with the same period when connected or not.
	 *
	 * @param timer
	 *            the timer to use.
	 * @param delay
	 *            the delay before the start.
	 * @param periodWhenAvaialble
	 *            the period for the polling when at least one interface is available.
	 * @param periodWhenLost
	 *            the period for the polling when all the interface are down.
	 * @param periodWhenInternet
	 *            the period for the polling when ther is internet access.
	 */
	public PollerConnectivityManager(Timer timer, long delay, long periodWhenAvaialble, long periodWhenLost,
			long periodWhenInternet) {
		this.networks = new HashMap<>();
		Network network = new Network(null);
		networks.put(EMPTY_NETWORK, network);
		this.timer = timer;
		this.periodWhenLost = periodWhenLost;
		this.periodWhenAvailable = periodWhenAvaialble;
		this.periodWhenInternet = periodWhenInternet;
		this.url = DEFAULT_URL;
		this.port = DEFAULT_PORT;
		this.activeNetwork = network;
		setActiveNetwork(network);
		setAvailable(false, network, getActiveNetworkInfo());
		startPolling(delay, periodWhenLost);
	}

	/**
	 * Starts the polling function.
	 *
	 * @param delay
	 *            the delay before starting.
	 */
	public void startPolling(long delay) {
		startPolling(delay, this.periodWhenLost);
	}

	/**
	 * Stops the polling function.
	 */
	public synchronized void stopPolling() {
		if (this.pollerTask != null) {
			this.pollerTask.cancel();
			this.pollerTask = null;
		}
	}

	/**
	 * Cancels the associated timer.
	 */
	public void cancel() {
		stopPolling();
		this.timer.cancel();
	}

	/**
	 * Logs a message.
	 *
	 * @param msg
	 *            the message to log.
	 */
	protected void log(String msg) {
		if (DEBUG) {
			System.out.println("[PollerConnectivityManager] INFO " + msg); //$NON-NLS-1$
		}
	}

	/**
	 * Logs a message and a throwable.
	 *
	 * @param msg
	 *            the message to log.
	 * @param t
	 *            a throwable to print.
	 */
	protected void log(String msg, Throwable t) {
		if (DEBUG) {
			System.out.println("[PollerConnectivityManager] INFO " + msg); //$NON-NLS-1$
			t.printStackTrace();
		}
	}

	private synchronized void startPolling(long delay, long period) {
		stopPolling();
		this.pollerTask = new ConnectivityManagerTimerTask();
		this.currentPeriod = period;
		this.timer.schedule(this.pollerTask, delay, currentPeriod);
	}

	private class ConnectivityManagerTimerTask extends TimerTask {

		@Override
		public void run() {
			log("Try connection"); //$NON-NLS-1$

			String activenetworkInterface = activeNetwork.getNetworkInterface();
			try {
				// If still connected to internet.
				if (activeNetwork.getCapabilities() == CONNECTED && activenetworkInterface != null
						&& getCapabilities(
								checkConnected(getNetworkInterface(activenetworkInterface))) == CONNECTED) {
					log("Skipped.");
					return;
				}
			} catch (SocketException e2) {
				// Sanity.
			}

			Enumeration<NetworkInterface> e;
			try {
				e = getNetworkInterfaces();
				checkNetworkInterfacesState(e);
			} catch (Throwable e1) {
				log("Error when polling", e1); //$NON-NLS-1$
			}
		}

		private void checkNetworkInterfacesState(Enumeration<NetworkInterface> interfaces) throws SocketException {
			Network activeNetwork = PollerConnectivityManager.this.activeNetwork;
			Network internetNetwork = null;
			Network availableNetwork = null;
			Network notAvailableNetwork = null;
			Network downNetwork = null;

			for (Entry<String, Network> network : networks.entrySet()) {
				network.getValue().setPresent(false);
			}

			if (interfaces != null) {
				while (interfaces.hasMoreElements()) {
					NetworkInterface n = interfaces.nextElement();
					if (!n.isLoopback()) {
						Network currentNetwork = getNetwork(n.getName());
						currentNetwork.setPresent(true);
						if (n.isUp()) {
							InetAddress connectedAddress = checkConnected(n);
							boolean available = connectedAddress != null;
							if (available) {
								NetworkCapabilities capabilities = getCapabilities(connectedAddress);
								if (capabilities == CONNECTED) {
									if (internetNetwork != activeNetwork || internetNetwork == null) {
										internetNetwork = currentNetwork;
										break;
									}
								} else if (availableNetwork != activeNetwork || availableNetwork == null) {
									availableNetwork = currentNetwork;
								}
							} else if (notAvailableNetwork != activeNetwork || notAvailableNetwork == null) {
								notAvailableNetwork = currentNetwork;
							}
						} else if (downNetwork != activeNetwork) {
							downNetwork = currentNetwork;
						}
					}
				}
			}

			if (internetNetwork == null) {
				for (Iterator<Entry<String, Network>> iterator = networks.entrySet().iterator(); iterator.hasNext();) {
					Entry<String, Network> entry = iterator.next();
					if (entry.getKey() != EMPTY_NETWORK && !entry.getValue().isPresent()) {
						iterator.remove();
					}
				}
			}

			updateNetworkState(internetNetwork, availableNetwork, notAvailableNetwork, downNetwork, activeNetwork);
		}

		private void updateNetworkState(Network internetNetwork, Network availableNetwork, Network notAvailableNetwork,
				Network downNetwork, Network activeNetwork) {
			updatePollingRate(availableNetwork != null, internetNetwork != null);
			// Set the best available network as active, if the current active is the same, it will be filtered
			// out during the set.
			if (internetNetwork != null) {
				notify(internetNetwork, true, CONNECTED, activeNetwork);
			} else if (availableNetwork != null) {
				notify(availableNetwork, true, UNCONNECTED, activeNetwork);
			} else if (notAvailableNetwork != null) {
				notify(notAvailableNetwork, false, UNCONNECTED, activeNetwork);
			} else if (downNetwork != null) {
				notify(downNetwork, false, UNCONNECTED, activeNetwork);
			} else {
				if (activeNetwork != networks.get(EMPTY_NETWORK)) {
					notify(networks.get(EMPTY_NETWORK), false, UNCONNECTED, activeNetwork);
				}
			}
		}

		private void notify(Network newNetwork, boolean available, NetworkCapabilities capabilities,
				Network previousNetwork) {
			boolean willAvailableChanged = newNetwork.willChanged(available);
			boolean willCapabilitiesChanged = newNetwork.willChanged(capabilities);
			newNetwork.setAvailable(available);
			newNetwork.setCapabilities(capabilities);
			activeNetwork = newNetwork;
			setActiveNetwork(newNetwork);
			if (newNetwork != previousNetwork) {
				// Notify previous owner.
				if (previousNetwork.getCapabilities() == CONNECTED) {
					previousNetwork.setCapabilities(UNCONNECTED);
					notifyNetworkCallbacks(previousNetwork, UNCONNECTED);
				}

				if (previousNetwork.isAvailable()) {
					previousNetwork.setAvailable(false);
					notifyNetworkCallbacks(previousNetwork, false);
				}
			}
			if (available) {
				if (willAvailableChanged) {
					notifyNetworkCallbacks(newNetwork, available);
				}
				if (willCapabilitiesChanged) {
					notifyNetworkCallbacks(newNetwork, capabilities);
				}
			} else if (newNetwork == previousNetwork) {
				if (willCapabilitiesChanged) {
					notifyNetworkCallbacks(newNetwork, capabilities);
				}
				if (willAvailableChanged) {
					notifyNetworkCallbacks(newNetwork, available);
				}
			}
		}

		private void updatePollingRate(boolean avaialble, boolean internet) {
			if (internet) {
				if (currentPeriod != periodWhenInternet) {
					startPolling(PollerConnectivityManager.this.periodWhenInternet,
							PollerConnectivityManager.this.periodWhenInternet);
				}
			} else if (avaialble && currentPeriod != periodWhenAvailable) {
				startPolling(PollerConnectivityManager.this.periodWhenAvailable,
						PollerConnectivityManager.this.periodWhenAvailable);
			} else if (!avaialble && currentPeriod != periodWhenLost) {
				startPolling(PollerConnectivityManager.this.periodWhenLost,
						PollerConnectivityManager.this.periodWhenLost);
			}
		}

		/**
		 * @param networkInterface
		 * @param networks
		 * @return
		 */
		private Network getNetwork(String networkInterface) {
			Network network = PollerConnectivityManager.this.networks.get(networkInterface);
			if (network == null) {
				network = new Network(networkInterface);
				PollerConnectivityManager.this.networks.put(networkInterface, network);
			}
			return network;
		}

		private InetAddress checkConnected(NetworkInterface n) {
			InetAddress connectedAddress = null;
			if (n != null) {
				Enumeration<InetAddress> ee = n.getInetAddresses();
				while (ee.hasMoreElements()) {
					InetAddress inetAddress = ee.nextElement();
					if (NetUtil.isValidInetAdress(inetAddress)) {
						connectedAddress = inetAddress;
						break;
					}
				}
			}
			return connectedAddress;
		}
	}

	/**
	 * Gets the capabilities of a network.
	 *
	 * @param connectedAddress
	 *            the address to get the capabilities from.
	 * @return the capabilities of the address.
	 */
	protected NetworkCapabilities getCapabilities(InetAddress connectedAddress) {
		NetworkCapabilities capabilities = UNCONNECTED;
		try (Socket socket = new Socket(InetAddress.getByName(PollerConnectivityManager.this.url),
				PollerConnectivityManager.this.port, connectedAddress, 0)) {
			capabilities = CONNECTED;
		} catch (IOException e) {
			// Sanity
		}
		return capabilities;
	}

	/**
	 * Isolate for testing.
	 *
	 * @return all the interface available.
	 * @throws SocketException
	 *             if an I/O exception occurs.
	 */
	protected Enumeration<NetworkInterface> getNetworkInterfaces() throws SocketException {
		return NetworkInterface.getNetworkInterfaces();
	}

	/**
	 * Isolate for testing.
	 * 
	 * @param name
	 *            the name of the interface.
	 * @return the interface with a name.
	 * @throws SocketException
	 *             if an I/O exception occurs.
	 */
	protected NetworkInterface getNetworkInterface(String name) throws SocketException {
		return NetworkInterface.getByName(name);
	}

}
