/*
 * Java
 *
 * Copyright 2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.microai;

/**
 * The <code>InputTensor</code> class offers services to deal with MicroAI input tensors.
 * <p>
 * <code>InputTensor</code> is an input of your {@link MLInferenceEngine}.
 */
public class InputTensor extends Tensor {
    private final int modelHandle;
    private final int tensorIndex;

    InputTensor(int modelHandle, int index) {
        super(modelHandle, index, true);
        this.modelHandle = modelHandle;
        this.tensorIndex = index;
    }

    /**
     * Loads the input data into the input tensor.
     * <p>
     * <code>inputData</code> is an array of signed or unsigned bytes.
     *
     * @param inputData
     *            the data to be loaded into the tensor.
     */
    public void setInputData(byte[] inputData) {
        MLNatives.setInputDataAsByteArray(this.modelHandle, this.tensorIndex, inputData);
    }

    /**
     * Loads the input data into the input tensor.
     * <p>
     * <code>inputData</code> is an array of signed or unsigned integer.
     *
     * @param inputData
     *            the data to be loaded into the tensor.
     */
    public void setInputData(int[] inputData) {
        MLNatives.setInputDataAsIntArray(this.modelHandle, this.tensorIndex, inputData);
    }

    /**
     * Loads the input data into the input tensor.
     * <p>
     * <code>inputData</code> is a float array.
     *
     * @param inputData
     *            the data to be loaded into the tensor.
     */
    public void setInputData(float[] inputData) {
        MLNatives.setInputDataAsFloatArray(this.modelHandle, this.tensorIndex, inputData);
    }
}
