/*
 * Java
 *
 * Copyright 2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.event;

import java.util.LinkedList;
import java.util.NoSuchElementException;

/**
 * The events queue is a mechanism to store some MicroUI and LLUIInput events in a linked list. These events will be
 * read by a MicroUI later.
 * <p>
 * The events are some <i>integers</i>. The format of this integer is producer / consumer dependent.
 */
public class EventQueue {

	private static EventQueue singleton;
	private final LinkedList<Integer> queue;

	private EventQueue() {
		this.queue = new LinkedList<>();
	}

	/**
	 * Gets the MicroUI events queue.
	 *
	 * @return the MicroUI events queue.
	 */
	public static synchronized EventQueue get() {
		if (singleton == null) {
			singleton = new EventQueue();
		}
		return singleton;
	}

	/**
	 * Adds the events in the queue of events, respecting the array order.
	 *
	 * @param events
	 *            the events to add.
	 */
	public synchronized void enqueudEvents(int[] events) {
		int l = events.length;
		for (int i = -1; ++i < l;) {
			add(events[i]);
		}
		notifyAll();
	}

	/**
	 * Adds an event in the queue of events.
	 *
	 * @param event
	 *            the event to add.
	 */
	public synchronized void enqueudEvent(int event) {
		add(event);
		notifyAll();
	}

	/**
	 * Reads an event in the queue of events. If there is no event, the thread is waiting while there is no event.
	 *
	 * @return the event read.
	 * @throws InterruptedException
	 *             if any thread interrupted the current thread before or while the current thread was waiting for a
	 *             notification.
	 */
	public synchronized int getEvent() throws InterruptedException {
		while (true) {
			try {
				return this.queue.remove().intValue();
			} catch (NoSuchElementException e) {
				wait();
			}
		}
	}

	/**
	 * Returns the number of elements in this list.
	 *
	 * @return the number of elements in this list
	 */
	public synchronized int size() {
		return this.queue.size();
	}

	/**
	 * Caller must be the owner of <code>this</code>.
	 *
	 * @param event
	 *            the event to add in queue.
	 */
	private void add(int event) {
		this.queue.offer(Integer.valueOf(event));
	}

}
