/*
 * Java
 *
 * Copyright 2018-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.aws.iot;

import javax.net.SocketFactory;
import javax.net.ssl.SSLSocketFactory;

import ej.bon.Util;

/**
 * This class represents the connection options of an AWS IoT Thing, needed to connect to the platform.
 */
public class AwsIotClientOptions {

	private static final String ID_PREFIX = "MicroEJ"; //$NON-NLS-1$

	private static final int DEFAULT_AWS_PORT = 8883;
	private static final int DEFAULT_CONNECTION_TIME_OUT = 30; // in seconds
	private static final int DEFAULT_CONNECTION_KEEP_ALIVE = 60; // in seconds

	private final String host;
	private final int port;
	private final String thingName;
	private final String clientId;
	private final int timeout;
	private final int keepAlive;
	private final SocketFactory socketFactory;
	private final String username;
	private final char[] password;

	/**
	 * Configure and Create an instance of {@link AwsIotClientOptions}.
	 */
	public static class Builder {
		private String host;
		private int port = DEFAULT_AWS_PORT;
		private String thingName;
		private String clientId;
		private int timeout = DEFAULT_CONNECTION_TIME_OUT;
		private int keepAlive = DEFAULT_CONNECTION_KEEP_ALIVE;
		private SocketFactory socketFactory = null;
		private String username;
		private char[] password;

		/**
		 *
		 * @return and instance of this builder class.
		 */
		public static Builder builder() {
			return new Builder();
		}

		/**
		 *
		 * @param host
		 *            AWS Iot Server host
		 * @return builder instance
		 */
		public Builder host(final String host) {
			this.host = host;
			return this;
		}

		/**
		 *
		 * @param port
		 *            AWS Iot Server Port. default is 8883
		 * @return builder instance
		 */
		public Builder port(final int port) {
			this.port = port;
			return this;
		}

		/**
		 *
		 * @param thingName
		 *            AWS Iot Thing Name
		 * @return builder instance
		 */
		public Builder thingName(final String thingName) {
			this.thingName = thingName;
			return this;
		}

		/**
		 *
		 * @param clientId
		 *            the MQTT ID. you can use {@link AwsIotClientOptions#generateClientId()} to get a random client ID.
		 * @return builder instance
		 */
		public Builder clientID(final String clientId) {
			this.clientId = clientId;
			return this;
		}

		/**
		 *
		 * @param timeout
		 *            MQTT client timeout in seconds. It must be {@code >} 0. default is 30 seconds
		 * @return builder instance
		 */
		public Builder timeout(final int timeout) {
			this.timeout = timeout;
			return this;
		}

		/**
		 *
		 * @param keepAlive
		 *            connection keep alive in seconds. It must be {@code >= 0}. default is 60
		 * @return builder instance
		 */
		public Builder keepAlive(final int keepAlive) {
			this.keepAlive = keepAlive;
			return this;
		}

		/**
		 *
		 * @param context
		 *            ssl context
		 * @return builder instance
		 */
		public Builder secure(final SocketFactory context) {
			this.socketFactory = context;
			return this;
		}

		/**
		 *
		 * @param username
		 *            to use in MQTT CONNECT
		 * @return builder instance
		 */
		public Builder username(final String username) {
			this.username = username;
			return this;
		}

		/**
		 *
		 * @param password
		 *            to use in MQTT CONNECT
		 * @return builder instance
		 */
		public Builder password(final char[] password) {
			this.password = password;
			return this;
		}

		/**
		 * Build option instance with values already set
		 *
		 * @return configured immutable instance of {@link AwsIotClientOptions}
		 */
		public AwsIotClientOptions build() {
			if (this.host == null || this.port == 0 || this.thingName == null || this.thingName.isEmpty()
					|| this.clientId == null || this.clientId.isEmpty() || this.timeout <= 0 || this.keepAlive <= 0) {
				throw new IllegalArgumentException();
			}

			return new AwsIotClientOptions(this.host, this.port, this.thingName, this.clientId, this.timeout,
					this.keepAlive, this.socketFactory, this.username, this.password);
		}

	}

	private AwsIotClientOptions(String serverHost, int serverPort, String thingId, String clientId, int timeout,
			int keepAlive, SocketFactory sslContext, String username, char[] password) { // NOSONAR internal API
		super();
		this.host = serverHost;
		this.port = serverPort;
		this.thingName = thingId;
		this.clientId = clientId;
		this.timeout = timeout;
		this.keepAlive = keepAlive;
		this.socketFactory = sslContext;
		this.username = username;
		this.password = password;
	}

	/**
	 * Gets the serverHost.
	 *
	 * @return the serverHost.
	 */
	public String getServerHost() {
		return this.host;
	}

	/**
	 * Gets the serverPort.
	 *
	 * @return the serverPort.
	 */
	public int getServerPort() {
		return this.port;
	}

	/**
	 * Gets the thingName.
	 *
	 * @return the thingName.
	 */
	public String getThingName() {
		return this.thingName;
	}

	/**
	 * Gets the clientId.
	 *
	 * @return the clientId.
	 */
	public String getClientId() {
		return this.clientId;
	}

	/**
	 * Gets the timeout.
	 *
	 * @return the timeout.
	 */
	public int getTimeout() {
		return this.timeout;
	}

	/**
	 * Gets the keepAlive.
	 *
	 * @return the keepAlive.
	 */
	public int getKeepAlive() {
		return this.keepAlive;
	}

	/**
	 * Gets the sslContext.
	 *
	 * @return the sslContext.
	 */
	public SocketFactory getSocketFactory() {
		return this.socketFactory;
	}

	/**
	 * Builds the server address.
	 *
	 * @return the correctly formatted server address
	 */
	public String getServerURL() {
		return (this.socketFactory instanceof SSLSocketFactory ? "ssl://" : "tcp://") + this.host + ":" + this.port; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	}

	/**
	 * generate a random client id id should be {@code <} 65535
	 *
	 * @return random client id prefixed by CLIENT_ID_PREFIX
	 */
	public static String generateClientId() {
		return ID_PREFIX + Util.platformTimeNanos();
	}

	/**
	 * Gets the username.
	 *
	 * @return the username.
	 */
	public String getUsername() {
		return this.username;
	}

	/**
	 * Gets the password.
	 *
	 * @return the password.
	 */
	public char[] getPassword() {
		return this.password == null ? null : this.password.clone();
	}
}
