/*
 * Java
 *
 * Copyright 2021-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg;

import com.is2t.hil.HIL;

import ej.microui.display.LLUIDisplay;
import ej.microui.display.LLUIPainter;
import ej.microui.display.MicroUIGraphicsContext;
import ej.microvg.paint.VGColor;
import ej.microvg.paint.VGPaint;

/**
 * This class provides implementations for the native methods of MicroVG library which perform some drawings.
 * <p>
 * It follows the rules described in {@link LLUIPainter}.
 */
public class LLVGPainter {

	/**
	 * Value returned on success.
	 */
	public static final int RETURN_SUCCESS = 0;

	/**
	 * Value returned when resource is closed
	 */
	public static final int RESOURCE_CLOSED = -6;

	/**
	 * Matrix to use when a null matrix is given in parameter (prevent checks on null)
	 */
	private static final float[] IDENTITY_MATRIX = MatrixHelper.createIdentity();

	private LLVGPainter() {
		// hide implicit constructor
	}

	/**
	 * Draws a path filled with the given color.
	 * <p>
	 * The specified path will be transformed by the given matrix and drawn into the specified target buffer using the
	 * supplied color.
	 *
	 * @param gcArray
	 *            the destination
	 * @param path
	 *            the path to draw
	 * @param x
	 *            the destination X coordinate
	 * @param y
	 *            the destination Y coordinate
	 * @param matrix
	 *            the deformation to apply
	 * @param fillRule
	 *            the fill type
	 * @param blend
	 *            the blend mode
	 * @param color
	 *            the color to apply (with an opacity)
	 */
	public static void drawPath(byte[] gcArray, byte[] path, int x, int y, float[] matrix, int fillRule, int blend,
			int color) {
		LLUIDisplay engine = LLUIDisplay.Instance;
		synchronized (engine) {
			MicroUIGraphicsContext gc = engine.mapMicroUIGraphicsContext(gcArray);
			if (gc.requestDrawing()) {
				LLVGEngine drawer = LLVGEngine.Instance;
				drawer.getVGDrawer(gc).drawPath(gc, drawer.mapPath(path), x, y, refreshMatrix(matrix), fillRule, blend,
						new VGColor(color));
			}
		}
	}

	/**
	 * Draws a path filled with a linear gradient.
	 * <p>
	 * The specified path will be transformed by the given matrix and filled by the transformed gradient pattern.
	 *
	 * @param gcArray
	 *            the destination
	 * @param path
	 *            the path to draw
	 * @param x
	 *            the destination X coordinate
	 * @param y
	 *            the destination Y coordinate
	 * @param matrix
	 *            the deformation to apply
	 * @param fillRule
	 *            the fill type
	 * @param alpha
	 *            the global opacity to apply
	 * @param blend
	 *            the blend mode
	 * @param gradientData
	 *            the gradient data
	 * @param gradientMatrix
	 *            the gradient matrix (can be null)
	 */
	public static void drawGradient(byte[] gcArray, byte[] path, int x, int y, float[] matrix, int fillRule, int alpha,
			int blend, int[] gradientData, float[] gradientMatrix) {

		LLUIDisplay engine = LLUIDisplay.Instance;
		synchronized (engine) {
			MicroUIGraphicsContext gc = engine.mapMicroUIGraphicsContext(gcArray);
			if (gc.requestDrawing()) {
				LLVGEngine vgEngine = LLVGEngine.Instance;
				VGPaint p = vgEngine.mapGradient(gradientData, gradientMatrix, matrix, alpha);
				vgEngine.getVGDrawer(gc).drawPath(gc, vgEngine.mapPath(path), x, y, refreshMatrix(matrix), fillRule,
						blend, p);
			}
		}
	}

	/**
	 * Draws a string with the color of the graphics context.
	 *
	 * @param gcArray
	 *            the graphics context to draw on
	 * @param string
	 *            the array of characters to draw
	 * @param face
	 *            the font face to use
	 * @param size
	 *            the font size, in pixels
	 * @param x
	 *            the destination X coordinate
	 * @param y
	 *            the destination Y coordinate
	 * @param matrix
	 *            the transformation matrix to apply (can be null)
	 * @param alpha
	 *            the global opacity rendering value
	 * @param blendMode
	 *            the blend mode to use
	 * @param letterSpacing
	 *            the extra letter spacing to use
	 * @return {@link #RETURN_SUCCESS} on success or an error code
	 */
	public static int drawString(byte[] gcArray, char[] string, int face, float size, float x, float y, float[] matrix,
			int alpha, int blendMode, float letterSpacing) {

		int ret = RETURN_SUCCESS;
		LLUIDisplay engine = LLUIDisplay.Instance;
		synchronized (engine) {
			MicroUIGraphicsContext gc = engine.mapMicroUIGraphicsContext(gcArray);
			if (gc.requestDrawing()) {
				ret = drawCharArray(gc, string, face, size, x, y, matrix, blendMode, letterSpacing, 0, 0, alpha);
			}
		}
		return ret;
	}

	/**
	 * Draws a string with a linear gradient.
	 *
	 * @param gcArray
	 *            the graphics context to draw on
	 * @param string
	 *            the array of characters to draw
	 * @param face
	 *            the font face to use
	 * @param size
	 *            the font size, in pixels
	 * @param x
	 *            the destination X coordinate
	 * @param y
	 *            the destination Y coordinate
	 * @param matrix
	 *            the transformation matrix to apply
	 * @param alpha
	 *            the global opacity rendering value
	 * @param blendMode
	 *            the blend mode to use
	 * @param letterSpacing
	 *            the extra letter spacing to use
	 * @param gradientData
	 *            the gradient to apply
	 * @param gradientMatrix
	 *            the gradient deformation, may be null (means identity)
	 * @return {@link #RETURN_SUCCESS} on success or an error code
	 */
	public static int drawGradientString(byte[] gcArray, char[] string, int face, float size, float x, float y,
			float[] matrix, int alpha, int blendMode, float letterSpacing, int[] gradientData, float[] gradientMatrix) {

		int ret = RETURN_SUCCESS;
		LLUIDisplay engine = LLUIDisplay.Instance;
		synchronized (engine) {
			MicroUIGraphicsContext gc = engine.mapMicroUIGraphicsContext(gcArray);
			if (gc.requestDrawing()) {
				ret = drawCharArray(gc, string, face, size, x, y, matrix, blendMode, letterSpacing, 0, 0, gradientData,
						gradientMatrix, alpha);
			}
		}
		return ret;
	}

	/**
	 * Draws a string along a circle, with the color of the graphics context.
	 *
	 * @param gcArray
	 *            the graphics context to draw on
	 * @param string
	 *            the array of characters to draw
	 * @param face
	 *            the font face to use
	 * @param size
	 *            the font size, in pixels
	 * @param x
	 *            the destination X coordinate
	 * @param y
	 *            the destination Y coordinate
	 * @param matrix
	 *            the transformation matrix to apply
	 * @param alpha
	 *            the global opacity rendering value
	 * @param blendMode
	 *            the blend mode to use
	 * @param letterSpacing
	 *            the extra letter spacing to use
	 * @param radius
	 *            the radius of the circle
	 * @param direction
	 *            the direction of the text along the circle
	 * @return {@link #RETURN_SUCCESS} on success or an error code
	 */
	public static int drawStringOnCircle(byte[] gcArray, char[] string, int face, float size, int x, int y,
			float[] matrix, int alpha, int blendMode, float letterSpacing, float radius, int direction) {

		int ret = RETURN_SUCCESS;
		LLUIDisplay engine = LLUIDisplay.Instance;
		synchronized (engine) {
			MicroUIGraphicsContext gc = engine.mapMicroUIGraphicsContext(gcArray);
			if (gc.requestDrawing()) {
				ret = drawCharArray(gc, string, face, size, x, y, matrix, blendMode, letterSpacing, radius, direction,
						alpha);
			}
		}
		return ret;
	}

	/**
	 * Draws a string along a circle, with a linear gradient.
	 *
	 * @param gcArray
	 *            the graphics context to draw on
	 * @param string
	 *            the array of characters to draw
	 * @param face
	 *            the font face to use
	 * @param size
	 *            the font size, in pixels
	 * @param x
	 *            the destination X coordinate
	 * @param y
	 *            the destination Y coordinate
	 * @param matrix
	 *            the transformation matrix to apply
	 * @param alpha
	 *            the global opacity rendering value
	 * @param blendMode
	 *            the blend mode to use
	 * @param letterSpacing
	 *            the extra letter spacing to use
	 * @param radius
	 *            the radius of the circle
	 * @param direction
	 *            the direction of the text along the circle
	 * @param gradientData
	 *            the gradient to apply
	 * @param gradientMatrix
	 *            the gradient deformation, may be null (means identity)
	 * @return {@link #RETURN_SUCCESS} on success or an error code
	 */
	public static int drawGradientStringOnCircle(byte[] gcArray, char[] string, int face, float size, int x, int y,
			float[] matrix, int alpha, int blendMode, float letterSpacing, float radius, int direction,
			int[] gradientData, float[] gradientMatrix) {

		int ret = RETURN_SUCCESS;
		LLUIDisplay engine = LLUIDisplay.Instance;
		synchronized (engine) {
			MicroUIGraphicsContext gc = engine.mapMicroUIGraphicsContext(gcArray);
			if (gc.requestDrawing()) {
				ret = drawCharArray(gc, string, face, size, x, y, matrix, blendMode, letterSpacing, radius, direction,
						gradientData, gradientMatrix, alpha);
			}
		}
		return ret;
	}

	/**
	 * Draws a string along a circle, with the color of the graphics context.
	 *
	 * @param gc
	 *            the destination
	 * @param string
	 *            the array of characters to draw
	 * @param face
	 *            the font face to use
	 * @param size
	 *            the font size, in pixels
	 * @param x
	 *            the destination X coordinate
	 * @param y
	 *            the destination Y coordinate
	 * @param matrix
	 *            the transformation matrix to apply
	 * @param blend
	 *            the blend mode to use
	 * @param letterSpacing
	 *            the extra letter spacing to use
	 * @param radius
	 *            the radius of the circle
	 * @param direction
	 *            the direction of the text along the circle
	 * @param alpha
	 *            the global opacity to apply
	 * @return {@link #RETURN_SUCCESS} on success or an error code
	 */
	private static int drawCharArray(MicroUIGraphicsContext gc, char[] string, int face, float size, float x, float y,
			float[] matrix, int blend, float letterSpacing, float radius, int direction, int alpha) {
		VGColor gcColor = new VGColor(VGPaint.applyOpacity(gc.getMicroUIColor(), alpha));
		refreshContent(string, 0, string.length);
		return LLVGEngine.Instance.getVGDrawer(gc).drawString(gc, string, face, size, x, y, refreshMatrix(matrix),
				blend, gcColor, letterSpacing, radius, direction);
	}

	/**
	 * Draws a string along a circle, with a linear gradient.
	 *
	 * @param gc
	 *            the destination
	 * @param string
	 *            the array of characters to draw
	 * @param face
	 *            the font face to use
	 * @param size
	 *            the font size, in pixels
	 * @param x
	 *            the destination X coordinate
	 * @param y
	 *            the destination Y coordinate
	 * @param matrix
	 *            the transformation matrix to apply
	 * @param blend
	 *            the blend mode to use
	 * @param letterSpacing
	 *            the extra letter spacing to use
	 * @param radius
	 *            the radius of the circle
	 * @param direction
	 *            the direction of the text along the circle
	 * @param gradientData
	 *            the gradient to apply
	 * @param gradientMatrix
	 *            the gradient deformation
	 * @param alpha
	 *            the global opacity rendering value
	 * @return {@link #RETURN_SUCCESS} on success or an error code
	 */
	private static int drawCharArray(MicroUIGraphicsContext gc, char[] string, int face, float size, float x, float y,
			float[] matrix, int blend, float letterSpacing, float radius, int direction, int[] gradientData,
			float[] gradientMatrix, int alpha) {
		LLVGEngine engine = LLVGEngine.Instance;
		VGPaint p = engine.mapGradient(gradientData, gradientMatrix, matrix, alpha);
		refreshContent(string, 0, string.length);
		return engine.getVGDrawer(gc).drawString(gc, string, face, size, x, y, refreshMatrix(matrix), blend, p,
				letterSpacing, radius, direction);
	}

	/**
	 * Draws an image with transformation and opacity. Optionally apply an animation and / or a color filtering.
	 *
	 * @param gcArray
	 *            the graphics context to draw on
	 * @param sniContext
	 *            the image to draw
	 * @param x
	 *            the destination X coordinate
	 * @param y
	 *            the destination Y coordinate
	 * @param matrix
	 *            the transformation matrix to apply
	 * @param alpha
	 *            the global opacity rendering value
	 * @param elapsedTime
	 *            the elapsed time since the beginning of the animation, in milliseconds
	 * @param colorMatrix
	 *            the color matrix used to transform colors
	 * @return {@link #RETURN_SUCCESS} on success or an error code
	 */
	public static int drawImage(byte[] gcArray, byte[] sniContext, int x, int y, float[] matrix, int alpha,
			long elapsedTime, float[] colorMatrix) {

		LLUIDisplay engine = LLUIDisplay.Instance;
		int ret = RETURN_SUCCESS;
		synchronized (engine) {
			MicroUIGraphicsContext gc = engine.mapMicroUIGraphicsContext(gcArray);
			if (gc.requestDrawing()) {

				refreshContent(colorMatrix);
				matrix = refreshMatrix(matrix);

				if (x != 0 || y != 0) {
					float[] temp = MatrixHelper.create(x, y);
					MatrixHelper.concatenate(temp, matrix);
					matrix = temp;
				}
				LLVGEngine vgEngine = LLVGEngine.Instance;
				MicroVGImage image = vgEngine.mapVectorImage(sniContext);

				if (null == image) {
					ret = RESOURCE_CLOSED;
				} else {
					ret = vgEngine.getVGDrawer(gc).drawImage(gc, vgEngine.mapVectorImage(sniContext), matrix, alpha,
							elapsedTime, colorMatrix);
				}
			}
		}
		return ret;
	}

	/**
	 * Refreshes the matrix given by the application. If the matrix is {@code null}, returns the unique instance of the
	 * identity matrix.
	 *
	 * @param matrix
	 *            the matrix to refresh
	 * @return the refreshed matrix or the identity matrix
	 */
	public static float[] refreshMatrix(float[] matrix) {
		if (null != matrix) {
			refreshContent(matrix, 0, 9);
			return matrix;
		}
		return IDENTITY_MATRIX;
	}

	/**
	 * Refreshes an array. The array can be {@code null} (nothing to refresh).
	 *
	 * @param array
	 *            the array to refresh
	 */
	public static void refreshContent(Object array) {
		if (null != array) {
			HIL.getInstance().refreshContent(array);
		}
	}

	/**
	 * Refreshes a range of an array.
	 *
	 * @param array
	 *            the array to refresh
	 * @param offset
	 *            the start index of the range
	 * @param length
	 *            the length of the range
	 */
	public static void refreshContent(Object array, int offset, int length) {
		HIL.getInstance().refreshContent(array, offset, length);
	}
}
