/*
 * Java
 *
 * Copyright 2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.hoka.http;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import ej.annotation.Nullable;
import ej.hoka.log.HokaLogger;

/**
 * Load and provide server configuration form hoka.properties on the classpath.
 */
public final class Config {

	private final Properties props = new Properties();

	private Config() {
		try {
			final InputStream is = this.getClass().getResourceAsStream("/hoka.properties"); //$NON-NLS-1$
			if (is != null) {
				this.props.load(is);
			}
		} catch (final IOException e) { // NOSONAR
			String msg = "config file 'hoka.properties' was not found. Default values will be used.\n" + e.getMessage(); //$NON-NLS-1$
			HokaLogger.instance.warn(msg);
		}

	}

	private static class Holder {
		public static final Config config = new Config();
	}

	/**
	 * Gets the configuration instance.
	 *
	 * @return the configuration instance
	 */
	public static Config getInstance() {
		return Holder.config;
	}

	/**
	 * Gets the buffer size.
	 *
	 * @return buffer size
	 */
	public int getBufferSize() {
		return Integer.parseInt(this.props.getProperty("hoka.buffer.size", "4096")); //$NON-NLS-1$ //$NON-NLS-2$
	}

	/**
	 * Gets the log level.
	 *
	 * @return log level
	 */
	public String getLoggerLevel() {
		return this.props.getProperty("hoka.logger.level", "INFO"); //$NON-NLS-1$ //$NON-NLS-2$
	}

	/**
	 * Gets the logger class name.
	 *
	 * @return logger class name
	 */
	@Nullable
	public String getLoggerClassName() {
		return this.props.getProperty("hoka.logger.class"); //$NON-NLS-1$
	}

}
