/*
 * Java
 *
 * Copyright 2009-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.hoka.http.requesthandler;

import static ej.hoka.http.HttpConstants.FSLASH;
import static ej.hoka.http.HttpConstants.HEADER_CACHE_CONTROLE;
import static ej.hoka.http.HttpConstants.HEADER_CONTENT_ENCODING;
import static ej.hoka.http.HttpConstants.HEADER_CONTENT_TYPE;
import static ej.hoka.http.HttpServer.halt;
import static ej.hoka.http.support.Mime.MIME_DEFAULT_BINARY;
import static ej.hoka.http.support.Mime.getMIMEType;

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;

import ej.annotation.Nullable;
import ej.hoka.http.HttpConstants;
import ej.hoka.http.support.Mime;
import ej.hoka.log.HokaLogger;

/**
 * Class path Files Handler .
 * <p>
 * Retrieves the URI of the request and tries to find a matching resource in the application class path.
 * <p>
 * Example:
 * <p>
 * Given the URI <code>http://192.168.1.1/my/wonderful/resource.html</code>, the Resource Request Handler, with root
 * directory <code>/my/package/</code> will try to find the resource <code>/my/package/my/wonderful/resource.html</code>
 * in the application's classpath (using {@link Class#getResourceAsStream(String)}).
 */
public class ClasspathFilesHandler implements StaticFilesHandler {

	private static final String GZIP_FILE_EXTENSION = ".gz"; //$NON-NLS-1$
	private static final String CONTENT_ENCODING_GZIP = "gzip"; //$NON-NLS-1$

	private static final String DEFAULT_INDEX = "index.html"; //$NON-NLS-1$

	private static final String DIRECTORY_TRAVERSAL_SEQUENCE = ".."; //$NON-NLS-1$

	private final String root;
	private final String index;
	private final int expireTimeSeconds;
	private final boolean guessMimeType;

	/**
	 * Constructs a resource request handler with given root directory path.
	 * <p>
	 * In case the requested resource is a directory, the <code>index</code> resource in this directory, if it exists,
	 * is sent.
	 *
	 *
	 * @param welcomeFile
	 *
	 * @param expirationTime
	 *
	 */
	private ClasspathFilesHandler(String rootDirectory, String welcomeFile, int expirationTime, boolean guessMimeType) {

		this.root = rootDirectory;
		this.index = welcomeFile;
		this.expireTimeSeconds = expirationTime;
		this.guessMimeType = guessMimeType;
	}

	/**
	 * Build an instance of {@link ClasspathFilesHandler}.
	 *
	 * @return instance of {@link ClasspathFilesHandlerBuilder}
	 */
	public static ClasspathFilesHandlerBuilder builder() {
		return new ClasspathFilesHandlerBuilder();
	}

	/**
	 * {@link ClasspathFilesHandler} builder.
	 */
	public static class ClasspathFilesHandlerBuilder {
		private String root = HttpConstants.EMPTY;
		private String index = DEFAULT_INDEX;
		private int expireTimeSeconds;
		private boolean guessMimeType = false;

		private ClasspathFilesHandlerBuilder() {
			// no-op
		}

		/**
		 * Returns the instance of {@link ClasspathFilesHandlerBuilder} configured with the given path as the root
		 * directory for resources to serve.
		 *
		 * @param root
		 *            the path of the root directory for resources to serve.
		 * @return the instance of {@link ClasspathFilesHandlerBuilder} to continue configuring the
		 *         {@link ClasspathFilesHandler} instance.
		 */
		public ClasspathFilesHandlerBuilder rootDirectory(String root) {
			this.root = root;
			if (this.root.endsWith(FSLASH)) {
				this.root = this.root.substring(0, this.root.length() - 1);
			}

			return this;
		}

		/**
		 * Returns the instance of {@link ClasspathFilesHandlerBuilder} configured with the given file name as the
		 * index.
		 *
		 * @param welcome
		 *            the directory index file name to serve in case a directory is requested.
		 * @return the instance of {@link ClasspathFilesHandlerBuilder} to continue configuring the
		 *         {@link ClasspathFilesHandler} instance.
		 */
		public ClasspathFilesHandlerBuilder welcomeFile(String welcome) {
			this.index = welcome;
			return this;
		}

		/**
		 * Returns the instance of {@link ClasspathFilesHandlerBuilder} configured with the given duration as the
		 * expiration time for static resources.
		 *
		 * @param expireTimeSeconds
		 *            Sets the expire-time in seconds for static resources.
		 * @return the instance of {@link ClasspathFilesHandlerBuilder} to continue configuring the
		 *         {@link ClasspathFilesHandler} instance.
		 */
		public ClasspathFilesHandlerBuilder expireTimeSeconds(int expireTimeSeconds) {
			this.expireTimeSeconds = expireTimeSeconds;
			return this;
		}

		/**
		 * activate content type guessing. the content-type header will added with the guessed type. More types can be
		 * registered using {@link Mime}. Note this may impact the performance.
		 *
		 * @return the instance of {@link ClasspathFilesHandlerBuilder} to continue configuring the
		 *         {@link ClasspathFilesHandler} instance.
		 */
		public ClasspathFilesHandlerBuilder guessMimeType() {
			this.guessMimeType = true;
			return this;
		}

		/**
		 * Builds and returns the {@link ClasspathFilesHandler} instance.
		 *
		 * @return create {@link ClasspathFilesHandler} instance.
		 */
		public ClasspathFilesHandler build() {
			return new ClasspathFilesHandler(this.root, this.index, this.expireTimeSeconds, this.guessMimeType);
		}

	}

	/**
	 * The generic behavior of this request handler implementation is to find a resource matching the given path in the
	 * classpath.
	 *
	 * @param path
	 *            file path
	 * @param headers
	 *            the response headers
	 */
	@Override
	@Nullable
	public InputStream serve(String path, Map<String, String> headers) {
		String uri = this.root + path;

		if (uri.contains(DIRECTORY_TRAVERSAL_SEQUENCE)) {
			// For security reasons, do not handle request to URI with a directory traversal sequence.
			halt(HttpConstants.HTTP_STATUS_FORBIDDEN);
		}

		if (uri.endsWith(FSLASH)) {
			uri = uri.substring(0, uri.length() - 1);
		}

		if (this.root.equals(uri)) {
			uri += FSLASH + this.index;
		}

		final InputStream resourceStream = getClass().getResourceAsStream(uri);
		if (resourceStream == null) {
			return null;
		}

		try {
			int contentLength = resourceStream.available();
			headers.put(HttpConstants.HEADER_CONTENT_LENGTH, "" + contentLength); //$NON-NLS-1$
		} catch (IOException e) { // NOSONAR
			// no-op ignore
			String msg = "error while getting resource size" + e.getMessage(); //$NON-NLS-1$
			HokaLogger.instance.trace(msg);
		}

		if (this.guessMimeType) {
			String mimeType = getMimeType(uri);
			headers.put(HEADER_CONTENT_TYPE, mimeType);
			if (CONTENT_ENCODING_GZIP.equals(mimeType)) {
				headers.put(HEADER_CONTENT_ENCODING, CONTENT_ENCODING_GZIP);
			}
		}

		if (this.expireTimeSeconds > 0) {
			headers.put(HEADER_CACHE_CONTROLE, "private, max-age=" + this.expireTimeSeconds); //$NON-NLS-1$
		}

		return resourceStream;
	}

	private String getMimeType(String uri) {
		String mimeType;
		if (uri.endsWith(GZIP_FILE_EXTENSION)) {
			mimeType = getMIMEType(uri.substring(0, uri.length() - GZIP_FILE_EXTENSION.length()));
		} else {
			mimeType = getMIMEType(uri);
		}
		if (mimeType == null) {
			mimeType = MIME_DEFAULT_BINARY;
		}
		return mimeType;
	}

}
