/*
 * Java
 *
 * Copyright 2020-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.microej.tool.ui.generator;

/**
 * An implementation of this interface is required to convert an image in a
 * binary format compatible with MicroUI Image API. The input image must be
 * pixel addressable.
 * <p>
 * The MicroUI Image RAW format accepts some optional characteristics: an
 * optional header (most of hardware decoders are MicroUI Image RAW format
 * compatible but can require a header to be able to recognize the image itself)
 * and / or a row stride different than the image width in pixels.
 * <p>
 * When the image output format is "DISPLAY" and the LCD pixel format is
 * "CUSTOM" (that means not standard like ARGB8888 or RGB565 for instance),
 * image generator requires the extension to encode each pixel in LCD pixel
 * format (see {@link #convertARGBColorToDisplayColor(int)}).
 */
public interface MicroUIRawImageGeneratorExtension {

	/**
	 * Tells if the extension is able to read the given image. If true, the
	 * extension should keep a reference on the image.
	 *
	 * @param image
	 *            the source image.
	 * @return true when the extension is able to read the source image.
	 */
	boolean canRead(Image image);

	/**
	 * Gets the current source image width in pixels.
	 *
	 * @return the image width.
	 */
	int getWidth();

	/**
	 * Gets the current source image height in pixels.
	 *
	 * @return the image height.
	 */
	int getHeight();

	/**
	 * Gets the ARGB8888 pixel color at given position.
	 *
	 * @param x
	 *            the pixel X-coordinate
	 * @param y
	 *            the pixel Y-coordinate
	 * @return the ARGB8888 color.
	 */
	int get(int x, int y);

	/**
	 * Returns the preferred row stride in pixels to respect in output image.
	 * Default implementation returns the default stride calculated by the image
	 * generator. If returned stride is lower than default stride, default stride is
	 * used.
	 *
	 * @param defaultStride
	 *            the stride calculated by the image generator (in pixels).
	 * @return the preferred row stride (in pixels).
	 */
	// if lower than default, default is used
	default int getStride(int defaultStride) {
		return defaultStride;
	}

	/**
	 * Returns a header the image generator has to store just before the pixels
	 * data. This header is often used when image is used at runtime by a hardware
	 * accelerator.
	 * <p>
	 * No padding is added between this header and the pixels array. If some padding
	 * is required, the returned array must contain it.
	 * <p>
	 * Image generator ensures header address will be aligned on header size itself.
	 * <p>
	 * By default, no header is added.
	 *
	 * @return null or a specific image header.
	 */
	default byte[] getOptionalHeader() {
		return null;
	}

	/**
	 * Called only when converting an image in same format than LCD format (image
	 * output format is "DISPLAY") and when this LCD format is not standard (like
	 * RGB565 or ARGB8888 for instance).
	 * <p>
	 * Converts the ARGB color into the LCD color format.
	 * <p>
	 * ARGB value is interpreted as a 32-bit ARGB color, where the eight less
	 * significant bits matches the blue component, the next eight bits matches the
	 * green component, the next eight bits matches the red component and the next
	 * eight bits matches the alpha level.
	 * <p>
	 * Note: the alpha level may be ignored if the display pixel representation does
	 * not hold the alpha level information.
	 *
	 * @param color
	 *            the ARGB color to convert
	 * @return the converted color
	 */
	default int convertARGBColorToDisplayColor(int color) {
		return color;
	}

}