/*
 * Java
 *
 * Copyright 2009-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt;

import com.is2t.tools.GeometryTools;

import ej.annotation.NonNull;
import ej.annotation.Nullable;
import ej.microui.display.GraphicsContext;
import ej.microui.util.EventHandler;

/**
 * Widget is the superclass of all the user interface objects.
 * <p>
 * There are a number of important concepts involving widgets:
 * <ul>
 * <li>Invalidation.
 * <p>
 * Whenever the state of a widget changes in a way that may affect the layout of the panel of which it is a part then
 * the hierarchy of the widget must be ask to be revalidated. This can be achieved by invoking {@link #invalidate()}
 * followed by {@link #revalidate()} on the widget.
 * <li>Validation.
 * <p>
 * Validation is the process laying out the widgets on a panel. This is performed by invoking {@link Panel#validate()},
 * which has the side-effect of performing any required repainting after validation. An application will normally invoke
 * {@link Panel#revalidate()} after making a set of changes to widgets.
 * <p>
 * The {@link #validate(int, int)} method should not normally be invoked by applications or widget implementations. It
 * is used by the <code>Panel</code> to propagate the validation request down through the widget hierarchy - widgets
 * with children propagate the call to their children.</li>
 * <li>Repainting.
 * <p>
 * Any widget can be asked to repaint itself by invoking {@link #repaint()}. If a widget has children it will ask them
 * to repaint. If the widget is transparent it will cause the relevant area of its parent to be repainted. Note that a
 * repaint request does not trigger validation, and the scope of the repainting that results from a call to
 * {@link #repaint()} will never exceed the widget itself, its children (recursively), and, if it is transparent, its
 * parent (recursively if the parent is also transparent).</li>
 * <li>Preferred size.
 * <p>
 * A widget's preferred size is its optimal displayed size given its content. Calling {@link #validate(int, int)} causes
 * the preferred size to be determined, normally with the assistance of a renderer. The preferred size is distinct from
 * the widget's actual size on the display (which may itself be subject to clipping).</li>
 * <li>Actual size.
 * <p>
 * The actual size is set by calling {@link #setSize(int, int)} or {@link #setBounds(int, int, int, int)}.
 * <p>
 * </li>
 * </ul>
 */
// Widget inherits CompositeView for Composite to use add/remove/get
public abstract class Widget implements Renderable {

	// Widget characteristics flags.
	private static final int ENABLED_SHIFT = 0;
	private static final int ENABLED_MASK = 0x1 << ENABLED_SHIFT;
	private static final int VISIBLE_SHIFT = 1;
	private static final int VISIBLE_MASK = 0x1 << VISIBLE_SHIFT;

	// Relative position.
	/* package */short x;
	/* package */short y;
	/* package */short width;
	/* package */short height;

	// widget hierarchy
	/* package */Panel panel;
	/* package */Composite parent;

	// widget controller
	@Nullable
	private EventHandler eventHandler;

	// Widget characteristics.
	/* package */boolean visible;
	private boolean enabled;

	// preferred size
	private short preferredWidth;
	private short preferredHeight;

	/**
	 * Creates a widget.
	 * <p>
	 * By default:
	 * <ul>
	 * <li>its bounds are <code>0</code>,</li>
	 * <li>it is visible,</li>
	 * <li>it is enabled.</li>
	 * </ul>
	 */
	public Widget() {
		// Visible & enabled by default.
		this.visible = true;
		this.enabled = true;
	}

	@Override
	public int getX() {
		return this.x;
	}

	@Override
	public int getY() {
		return this.y;
	}

	@Override
	public int getWidth() {
		return this.width;
	}

	@Override
	public int getHeight() {
		return this.height;
	}

	/**
	 * Gets the preferred width of the widget.
	 * <p>
	 * The result returned is meaningful only if {@link #isValid()} is <code>true</code> or if
	 * {@link #setPreferredSize(int, int)} has been called explicitly.
	 *
	 * @return the preferred width of the widget.
	 */
	public int getPreferredWidth() {
		return this.preferredWidth;
	}

	/**
	 * Gets the preferred height of the widget.
	 * <p>
	 * The result returned is meaningful only if {@link #isValid()} is <code>true</code> or if
	 * {@link #setPreferredSize(int, int)} has been called explicitly.
	 *
	 * @return the preferred height of the widget.
	 */
	public int getPreferredHeight() {
		return this.preferredHeight;
	}

	/**
	 * Sets the preferred size of the widget.
	 *
	 * @param preferredWidth
	 *            the width to set.
	 * @param preferredHeight
	 *            the height to set.
	 */
	public void setPreferredSize(int preferredWidth, int preferredHeight) {
		this.preferredWidth = (short) preferredWidth;
		this.preferredHeight = (short) preferredHeight;
	}

	/**
	 * Sets the location of this widget.
	 *
	 * @param x
	 *            the x coordinate to set.
	 * @param y
	 *            the y coordinate to set.
	 */
	public void setLocation(int x, int y) {
		setBounds(x, y, this.width, this.height);
	}

	/**
	 * Sets the size of this widget.
	 *
	 * @param width
	 *            the width to set.
	 * @param height
	 *            the height to set.
	 */
	public void setSize(int width, int height) {
		setBounds(this.x, this.y, width, height);
	}

	/**
	 * Sets the bounds of this widget.
	 *
	 * @param x
	 *            the x coordinate to set.
	 * @param y
	 *            the y coordinate to set.
	 * @param width
	 *            the width to set.
	 * @param height
	 *            the height to set.
	 */
	public void setBounds(int x, int y, int width, int height) {
		simpleUdpate(x, y, Math.max(0, width), Math.max(0, height));
	}

	/*
	 * width and height MUST be positive
	 */
	private void simpleUdpate(int x, int y, int width, int height) {
		this.x = (short) x;
		this.y = (short) y;
		this.width = (short) width;
		this.height = (short) height;
	}

	@Override
	public void showNotify() {
		// do nothing by default
	}

	@Override
	public void hideNotify() {
		// do nothing by default
	}

	/**
	 * Gets the absolute x coordinate of the widget. That is, the x coordinate relative to the origin of the display.
	 *
	 * @return the absolute x coordinate of the widget.
	 * @throws IllegalArgumentException
	 *             if the widget is not connected to a {@link Panel}.
	 */
	public int getAbsoluteX() throws IllegalArgumentException {
		if (this.panel == null) {
			throw new IllegalArgumentException();
		}
		return getAbsoluteXInternal();
	}

	/* package */ int getAbsoluteXInternal() {
		int absoluteX = this.x;
		Composite parent = getParent();
		if (parent != null) {
			absoluteX += parent.getAbsoluteXInternal();
		} else {
			Panel panel = getPanel();
			if (panel != null) {
				absoluteX += panel.x;
			}
		}
		return absoluteX;
	}

	/**
	 * Gets the absolute y coordinate of the widget. That is, the y coordinate relative to the origin of the display.
	 *
	 * @return the absolute y coordinate of the widget.
	 * @throws IllegalArgumentException
	 *             if the widget is not connected to a {@link Panel}.
	 */
	public int getAbsoluteY() throws IllegalArgumentException {
		if (this.panel == null) {
			throw new IllegalArgumentException();
		}
		return getAbsoluteYInternal();
	}

	/* package */ int getAbsoluteYInternal() {
		int absoluteY = this.y;
		Composite parent = getParent();
		if (parent != null) {
			absoluteY += parent.getAbsoluteYInternal();
		} else {
			Panel panel = getPanel();
			if (panel != null) {
				absoluteY += panel.y;
			}
		}
		return absoluteY;
	}

	/**
	 * Gets the x coordinate relative to the widget computed from the given absolute x coordinate.
	 *
	 * @param absoluteX
	 *            the absolute x coordinate to convert.
	 * @return the widget relative x coordinate.
	 * @throws IllegalArgumentException
	 *             if the widget is not connected to a {@link Panel}.
	 * @since 1.0
	 */
	public int getRelativeX(int absoluteX) throws IllegalArgumentException {
		return absoluteX - getAbsoluteX();
	}

	/**
	 * Gets the y coordinate relative to the widget computed from the given absolute y coordinate.
	 *
	 * @param absoluteY
	 *            the absolute y coordinate to convert.
	 * @return the widget relative y coordinate.
	 * @throws IllegalArgumentException
	 *             if the widget is not connected to a {@link Panel}.
	 * @since 1.0
	 */
	public int getRelativeY(int absoluteY) throws IllegalArgumentException {
		return absoluteY - getAbsoluteY();
	}

	/**
	 * Gets the absolute x coordinate computed from the given x coordinate relative to the widget.
	 *
	 * @param relativeX
	 *            the widget relative x coordinate to convert.
	 * @return the absolute x coordinate.
	 * @since 1.0
	 */
	public int getAbsoluteX(int relativeX) {
		return relativeX + getAbsoluteX();
	}

	/**
	 * Gets the absolute y coordinate computed from the given y coordinate relative to the widget.
	 *
	 * @param relativeY
	 *            the widget relative y coordinate to convert.
	 * @return the absolute y coordinate.
	 * @since 1.0
	 */
	public int getAbsoluteY(int relativeY) {
		return relativeY + getAbsoluteY();
	}

	/**
	 * Gets whether this widget is visible or not.
	 *
	 * @return <code>true</code> if this widget is visible, <code>false</code> otherwise.
	 * @see #setVisible(boolean)
	 * @since 1.0
	 */
	public boolean isVisible() {
		return this.visible;
	}

	/**
	 * Sets this widget visible or not.
	 * <p>
	 * Declaring a widget as invisible means that it does not appear on the screen and its size its size will be set to
	 * <code>(0, 0)</code> during the validation.
	 * <p>
	 * If the widget is on a panel hierarchy, it is invalidated.
	 *
	 * @param visible
	 *            <code>true</code> to set this widget visible, <code>false</code> otherwise.
	 * @see #isValid()
	 * @see #invalidate()
	 * @since 1.0
	 */
	public void setVisible(boolean visible) {
		this.visible = visible;
		Panel panel = this.panel; // avoid sync
		if (isShown() && !panel.isValidating()) {
			invalidate();
		}
	}

	/**
	 * Gets whether this widget is transparent or not.
	 * <p>
	 * By default, a widget is transparent.
	 * <p>
	 * A transparent widget means that it will not repaint ALL the rectangular zone defined by its bounds. Then each
	 * time it needs to be repainted, its parent (recursively if also transparent) will be repainted within the bounds
	 * of the widget.<br>
	 * Each time a non-transparent widget needs to be repainted, it is the only one to be repainted.
	 *
	 * @return <code>true</code> if this widget is transparent, <code>false</code> otherwise.
	 * @see #contains(int, int)
	 */
	public boolean isTransparent() {
		return true;
	}

	/**
	 * Gets whether or not a location (x,y) is in the widget's bounds.
	 * <p>
	 * The given location is considered here as a relative location to parent.
	 *
	 * @param x
	 *            x coordinate.
	 * @param y
	 *            y coordinate.
	 * @return <code>true</code> if the <code>(x,y)</code> location is in widget bounds, <code>false</code> otherwise.
	 * @see #isTransparent()
	 */
	public boolean contains(int x, int y) {
		// copied in Panel
		return GeometryTools.contains(this.x, this.y, this.width, this.height, x, y);
	}

	/**
	 * Gets the widget at the specified location.
	 * <p>
	 * If this widget does not <code>contains(x, y)</code>, <code>null</code> is returned, else this widget is returned.
	 * The location is considered here as a relative location to parent.
	 *
	 * @param x
	 *            x coordinate.
	 * @param y
	 *            y coordinate.
	 * @return this widget if it <code>contains(x, y)</code>, <code>null</code> otherwise.
	 */
	@Nullable
	public Widget getWidgetAt(int x, int y) {
		if (contains(x, y)) {
			return this;
		}
		return null;
	}

	/**
	 * Gets whether or not this widget is the focus owner of its panel and its panel is the active one.
	 * <p>
	 * Gets <code>false</code> if this widget is not on a panel.
	 *
	 * @return <code>true</code> if this widget is the focus owner, <code>false</code> otherwise.
	 * @see Panel#isActive()
	 */
	public boolean hasFocus() {
		try {
			return this.panel.isActive() && isFocusOwner();
		} catch (NullPointerException e) {
			// not on a panel
			return false;
		}
	}

	boolean isFocusOwner() {
		try {
			return this.panel.getFocus() == this;
		} catch (NullPointerException e) {
			// Not on a panel.
			return false;
		}
	}

	/**
	 * Sets this widget as the focus owner of its panel if it is enabled.
	 * <p>
	 * If the widget is not in a panel hierarchy, nothing is done.
	 */
	public void requestFocus() {
		try {
			this.panel.setFocus(this);
		} catch (NullPointerException e) {
			// not on a panel
		}
	}

	/**
	 * Sets this widget as the focus owner of its panel if it is enabled.
	 * <p>
	 * If the widget is not in a panel hierarchy, nothing is done. <br>
	 * The given direction must be one of {@link MWT#UP}, {@link MWT#DOWN}, {@link MWT#LEFT}, {@link MWT#RIGHT}. <br>
	 * Composed widgets can override this method in order to manage internal focus.
	 *
	 * @param direction
	 *            the direction followed by the focus
	 * @return <code>true</code> if the widget take the focus, <code>false</code> otherwise.
	 */
	public boolean requestFocus(int direction) {
		if (isEnabled()) {
			requestFocus();
			return true;
		}
		return false;
	}

	/**
	 * Notifies the widget that it is now the focus owner of its panel.
	 * <p>
	 * The subclasses can override this method to add behavior.
	 */
	public void gainFocus() {
		// Do nothing.
	}

	/**
	 * Notifies the widget that it is no longer the focus owner of its panel.
	 * <p>
	 * The subclasses can override this method to add behavior.
	 */
	public void lostFocus() {
		// Do nothing.
	}

	/**
	 * Gets whether or not this widget is enabled.
	 * <p>
	 * A widget that is not enabled cannot get focus.<br>
	 * A widget that is not visible is also disabled.
	 *
	 * @return <code>true</code> if this widget is enabled, <code>false</code> otherwise.
	 */
	public boolean isEnabled() {
		return this.visible && this.enabled;
	}

	/**
	 * Sets this widget to be enabled or not. A widget must be enabled in order to receive focus, to be the subject of
	 * mouse focus, or to receive events.
	 * <p>
	 * Requests a repaint of the widget.
	 *
	 * @param enabled
	 *            <code>true</code> if this widget is to be enabled, <code>false</code> otherwise.
	 * @see Widget#repaint()
	 */
	public void setEnabled(boolean enabled) {
		boolean wasEnabled = this.enabled;
		if (wasEnabled != enabled) {
			this.enabled = enabled;
			repaint();
		}
	}

	/**
	 * Declares that the panel containing this widget needs to be laid out.
	 *
	 * @see Panel#invalidate()
	 * @since 2.0
	 */
	public void invalidate() {
		Panel panel = this.panel; // avoid sync
		if (panel != null) {
			panel.invalidate();
		}
	}

	/**
	 * Lays out all the hierarchy of the panel containing this widget if exists.
	 * <p>
	 * If the hierarchy is not declared as invalid, nothing is done.
	 *
	 * @see Panel#revalidate()
	 * @see #isValid()
	 * @since 1.0
	 */
	public void revalidate() {
		Panel panel = this.panel; // avoid sync
		if (panel != null) {
			panel.revalidate();
		}
	}

	/**
	 * Lays out all the widgets in the sub hierarchy of this widget.
	 * <p>
	 * It calls the methods {@link #validate(int, int)} and {@link #setBounds(int, int, int, int)} asynchronously
	 * (passing the current bounds). Therefore this method does not block until the validation of the hierarchy is done.
	 * <p>
	 * The validation is done even if the hierarchy is not declared invalid. The hierarchy remains invalid until a full
	 * validation is done.
	 *
	 * @see #isValid()
	 * @since 2.2
	 */
	public void revalidateSubTree() {
		final Panel panel = this.panel;
		if (panel != null) {
			Desktop desktop = panel.getDesktop();
			if (desktop != null && desktop.isShown()) {
				desktop.getDisplay().callSerially(new Runnable() {
					@Override
					public void run() {
						panel.startValidating();
						Widget widget = Widget.this;
						short width = widget.width;
						short height = widget.height;
						validate(width, height);
						setBounds(widget.x, widget.y, width, height);
						panel.stopValidating();
						repaint();
					}
				});
			}
		}
	}

	/**
	 * Lays out this widget.
	 * <p>
	 * After this call the preferred size will have been established.
	 * <p>
	 * If the widget is not visible, its preferred size is set to <code>(0, 0)</code>.
	 * <p>
	 * The parameters defines the maximum size available for this widget, or {@link MWT#NONE} if there is no constraint.
	 *
	 * @param widthHint
	 *            the width available for this widget or {@link MWT#NONE}.
	 * @param heightHint
	 *            the height available for this widget or {@link MWT#NONE}.
	 * @see #isVisible()
	 * @see #setPreferredSize(int, int)
	 */
	public abstract void validate(int widthHint, int heightHint);

	/**
	 * Gets whether this widget is valid.
	 * <p>
	 * A widget is valid if all of these conditions are meet:
	 * <ul>
	 * <li>if it is in a panel hierarchy,</li>
	 * <li>if it is visible,</li>
	 * <li>if all its panel hierarchy is valid.</li>
	 * </ul>
	 *
	 * @return <code>true</code> if this widget is valid, <code>false</code> otherwise.
	 * @see #isVisible()
	 * @see Panel#isValid()
	 */
	public boolean isValid() {
		if (this.panel != null) {
			return this.visible && this.panel.isValid();
		}
		return false;
	}

	/**
	 * Gets the parent of this widget or <code>null</code> if the widget is not in a hierarchy.
	 *
	 * @return the parent of this widget or <code>null</code>.
	 */
	@Nullable
	public Composite getParent() {
		return this.parent;
	}

	/**
	 * Gets the panel of this widget or <code>null</code> if the widget is not in a panel.
	 *
	 * @return the panel of this widget or <code>null</code>.
	 */
	@Nullable
	public Panel getPanel() {
		return this.panel;
	}

	/**
	 * Gets whether or not the widget is shown on a shown panel.
	 *
	 * @return <code>true</code> if the widget is shown, <code>false</code> otherwise.
	 * @see #getPanel()
	 * @see Panel#isShown()
	 */
	@Override
	public boolean isShown() {
		Panel panel = this.panel; // avoid sync
		return panel != null && panel.isShown();
	}

	/**
	 * Requests a repaint of this entire widget. This method returns immediately, the repaint of the widget is performed
	 * asynchronously.
	 * <p>
	 * If the widget is not shown, nothing is done.
	 * <p>
	 * If the widget is transparent, it requests a repaint of its parent within the widget's bounds.
	 */
	@Override
	public void repaint() {
		repaintInternal(0, 0, this.width, this.height);
	}

	/**
	 * Requests a repaint of a zone of this widget. This method returns immediately, the repaint of the widget is
	 * performed asynchronously.
	 * <p>
	 * If the widget is not shown, nothing is done.
	 * <p>
	 * If the widget is transparent, it requests a repaint of its parent within the requested bounds.
	 *
	 * @param x
	 *            the relative x coordinate of the area to repaint.
	 * @param y
	 *            the relative y coordinate of the area to repaint.
	 * @param width
	 *            the width of the area to repaint.
	 * @param height
	 *            the height of the area to repaint.
	 */
	@Override
	public void repaint(int x, int y, int width, int height) {
		// copied in Panel.repaint(int, int, int, int)
		// crop to widget size
		// must be done in this order: location before size
		if (x < 0) {
			width += x;
			x = 0;
		}
		if (y < 0) {
			height += y;
			y = 0;
		}
		width = Math.min(this.width - x, width);
		height = Math.min(this.height - y, height);
		if (width <= 0 || height <= 0) {
			return; // nothing to repaint
		}

		repaintInternal(x, y, width, height);
	}

	@Override
	public void setEventHandler(@Nullable EventHandler eventHandler) {
		this.eventHandler = eventHandler;
	}

	@Override
	@Nullable
	public EventHandler getEventHandler() {
		return this.eventHandler;
	}

	/**
	 * Called by the system if the widget is the owner of the focus of the active panel.
	 * <p>
	 * If an event handler is registered, its {@link EventHandler#handleEvent(int)} method is called.<br>
	 * Otherwise, do nothing and return <code>false</code> (do not consume event).
	 *
	 * @param event
	 *            the event to handle.
	 * @return <code>true</code> if the widget has consumed the event, <code>false</code> otherwise.
	 * @see #setEventHandler(EventHandler)
	 */
	@Override
	public boolean handleEvent(int event) {
		return RenderableHelper.handleEvent(event, this.eventHandler);
	}

	/* NOT IN API */

	/**
	 * Sets widget's parent and panel. The widget must not be in another composite or panel before this call.
	 */
	/* package */void setParent(@Nullable Composite parent) {
		setPanel(parent == null ? null : parent.panel);
		this.parent = parent;
	}

	/**
	 * Sets widget's panel. The widget must not be in another panel before this call.
	 */
	/* package */void setPanel(@Nullable Panel panel) {
		Panel oldPanel = this.panel; // avoid synchro
		if (panel != null) {
			// Sets the panel before notifying the show.
			setPanelOnly(panel);
			// check already in a panel hierarchy
			if (oldPanel == null && panel.isShown()) {
				showNotify();
			}
		} else {
			if (oldPanel != null && oldPanel.isShown()) {
				hideNotify();
			}
			// Removes the panel after notifying the hide.
			setPanelOnly(null);
		}
	}

	/* package */ void setPanelOnly(@Nullable Panel panel) {
		this.panel = panel;
	}

	/**
	 * Paints the widget.
	 *
	 * @see RenderableHelper#paintRenderable(GraphicsContext, Desktop, Renderable)
	 */
	/* package */void paint(@NonNull GraphicsContext g) {
		Panel panel = this.panel;
		if (panel != null) {
			Desktop desktop = panel.getDesktop();
			RenderableHelper.paintRenderable(g, desktop, this);
		}
	}

	/* package */void paint(@NonNull GraphicsContext g, int translateX, int translateY, int x, int y, int width,
			int height) {
		if (!this.visible) {
			return;
		}
		beforePaint(g, translateX, translateY, x, y, width, height);
		paint(g);
	}

	/**
	 * Initialize the graphics context before calling the component view paint() method (clip & translation), to paint
	 * only in the wanted area. x, y coordinates must be relatives.
	 */
	/* package */void beforePaint(@NonNull GraphicsContext g, int translateX, int translateY, int x, int y, int width,
			int height) {
		g.translate(-g.getTranslateX(), -g.getTranslateY());
		g.setSystemClip(translateX + x, translateY + y, width, height);
		g.translate(translateX + this.x, translateY + this.y);
		g.setClipAndUpdateSystemClip(0, 0, this.width, this.height);
	}

	/**
	 * Repaints the widget.<br>
	 * <ol>
	 * <li>If it is transparent, repaints its parent (composite or panel).</li>
	 * <li>If its panel is on the active one (on the top of the stacking order), repaints this widget only.</li>
	 * <li>Otherwise, see {@link Desktop#repaintFrom(Renderable, int, int, int, int)}.</li>
	 * </ol>
	 * <p>
	 * Nothing is done if the widget is not included in any hierarchy.
	 */
	private void repaintInternal(int x, int y, int width, int height) {
		if (!this.visible) {
			return;
		}
		Panel panel = this.panel;
		if (panel == null) {
			return; // not in a widget hierarchy, no need to repaint at all!
		}
		if (isTransparent()) {
			Renderable target = this.parent;
			if (target == null) {
				target = panel;
			}
			// repaint upward
			target.repaint(this.x + x, this.y + y, width, height);
		} else {
			Desktop desktop = panel.getDesktop();
			if (desktop != null && desktop.isShown()) {
				if (desktop.getActivePanel() == panel) {
					// on the top
					desktop.getDisplay().callSerially(new RepaintRenderable(desktop, x, y, width, height) {
						@Override
						protected void paint(@NonNull GraphicsContext g) {
							// Compute
							g.translate(-g.getTranslateX(), -g.getTranslateY());
							int leftX = Widget.this.getAbsoluteXInternal() + this.x;
							int topY = Widget.this.getAbsoluteYInternal() + this.y;
							g.setSystemClip(leftX, topY, this.width, this.height);
							g.setClip(leftX, topY, this.width, this.height);
							clipRecursive(g);
							Widget.this.paint(g, g.getTranslateX(), g.getTranslateY(), g.getClipX(), g.getClipY(),
									g.getClipWidth(), g.getClipHeight());
						}
					});
				} else {
					// not active, must repaint all renderables above
					desktop.repaintFrom(this, getAbsoluteXInternal() + x, getAbsoluteYInternal() + y, width, height);
				}
			}
		}
	}

	/* package */ void clip(@NonNull GraphicsContext g) {
		clipRecursive(g);
		g.translate(this.x, this.y);
		g.clipRect(0, 0, this.width, this.height);
	}

	/* package */ void clipRecursive(@NonNull GraphicsContext g) {
		Composite parent = getParent();
		if (parent != null) {
			parent.clip(g);
		} else {
			Panel panel = getPanel();
			if (panel != null) {
				g.translate(panel.x, panel.y);
				g.clipRect(0, 0, panel.width, panel.height);
			}
		}

	}

}
